from aiohttp import web
from aiohttp.web_exceptions import HTTPBadRequest
from aiohttp.web_response import json_response
from aiohttp_apispec import docs, querystring_schema, request_schema, response_schema
from marshmallow import Schema, fields, validate

from travel.avia.subscriptions.app.api.app import App
from travel.avia.subscriptions.app.api.exceptions import (
    InvalidEmail, InvalidTravelVertical, InvalidUserAuthType,
    NoAccess, EmailNotFound, UnknownPointsError
)
from travel.avia.subscriptions.app.api.schemas import (
    UserSubscriptionGetResponseSchema, UserSubscriptionGetRequestQueryStringSchema,
    BaseUserSubscriptionPutRequestSchema, PutStatsSchema, UserPromoSubscriptionDeleteRequestSchema
)
from travel.avia.subscriptions.app.lib.qkey import QkeyValidationError
from travel.avia.subscriptions.app.model.schemas import FilterSchema, MinPriceSchema
from travel.avia.subscriptions.app.model.storage import UpsertAction


class UserPriceChangeSubscriptionPutRequestSchema(BaseUserSubscriptionPutRequestSchema):
    qkey = fields.String(
        required=False,
        example='c213_c2_2017-12-21_2018-01-05_economy_1_0_1_ru',
    )
    qid = fields.String(
        required=False,
        example=(
            '171206-144220-171.ticket.plane.'
            'c54_c239_2017-12-28_2018-01-04_economy_1_0_0_ru.'
            'ru'
        )
    )
    filter_ = fields.Nested(FilterSchema, required=False, data_key='filter')
    date_range = fields.Integer(
        required=False,
        missing=1,
        validate=validate.Range(min=1, max=120)
    )
    min_price = fields.Nested(MinPriceSchema, required=False)


class UserPriceChangeSubscriptionDeleteResponseSchema(Schema):
    qkey = fields.String(example='c213_c2_2017-12-21_2018-01-05_economy_1_0_1_ru')
    unsubscription_success = fields.Bool()


def get_user_price_change_subscription_view(app: App):
    class UserPriceChangeSubscriptionView(web.View):
        @docs(
            tags=['user_price_change_subscription'],
            summary='Subscribes a user to a price change subscription',
            description='Subscribes a user to a price change subscription by his email, '
                        'credentials (session, passport, iCookie), and other meta parameters',
        )
        @request_schema(UserPriceChangeSubscriptionPutRequestSchema)
        @response_schema(
            PutStatsSchema,
            description='Number of created and updated subscriptions in a database.'
                        ' Depens on number of credentials passed.'
        )
        async def put(self):
            data = self.request['data']
            try:
                result = await app.user_price_change_subscription_actor.put(**data)
                return json_response(
                    {'updated': len(result[UpsertAction.UPDATE]), 'created': len(result[UpsertAction.INSERT])}
                )
            except (
                QkeyValidationError, InvalidEmail, InvalidTravelVertical,
                InvalidUserAuthType, UnknownPointsError
            ) as e:
                return HTTPBadRequest(text='{}'.format(e))

        @docs(
            tags=['user_price_change_subscription'],
            summary='Get a list of user subscriptions',
            description='Get a list of user subscriptions, '
                        'identified by email and credentials (session, passport, iCookie).'
                        'Example: `/email=user@domain.com&auth_type=session&auth_value=123` will list all subscriptions'
                        'for user@domain.com that were created using unauthorized session "123".'
                        '\n'
                        'Email parameter is optional and is used as a filter.',
        )
        @querystring_schema(UserSubscriptionGetRequestQueryStringSchema)
        @response_schema(UserSubscriptionGetResponseSchema(many=True))
        async def get(self):
            try:
                data = {
                    'email': self.request['querystring'].get('email'),
                    'credentials': [
                        (
                            self.request['querystring']['auth_type'],
                            self.request['querystring']['auth_value'],
                        ),
                    ],
                }
                result = await app.user_price_change_subscription_actor.get_subscriptions_list(
                    **data, raise_on_auth_type_absent=True
                )
                return json_response(list(result))
            except InvalidUserAuthType as e:
                return HTTPBadRequest(text='{}'.format(e))

        @docs(
            tags=['user_price_change_subscription'],
            summary='Delete subscriptions of user',
            description='Delete subscriptions of user by his email, '
                        'credentials (session, passport, iCookie), and other meta parameters',
        )
        @request_schema(UserPromoSubscriptionDeleteRequestSchema)
        @response_schema(UserPriceChangeSubscriptionDeleteResponseSchema(many=True))
        async def delete(self):
            data = self.request['data']
            try:
                subscription_names = await app.user_price_change_subscription_actor.delete(**data)
                result = []

                for subscription in data['subscription_codes']:
                    result.append({
                        'qkey': subscription,
                        'unsubscription_success': subscription in subscription_names
                    })

                return json_response(result)
            except (NoAccess, EmailNotFound) as e:
                return HTTPBadRequest(text='{}'.format(e))

    return UserPriceChangeSubscriptionView
