# coding=utf-8
from __future__ import unicode_literals

from aiohttp import web
from aiohttp.web_exceptions import HTTPBadRequest
from aiohttp.web_response import json_response
from aiohttp_apispec import (docs, querystring_schema, request_schema, response_schema)
from marshmallow import Schema, fields

from travel.avia.subscriptions.app.api.app import App
from travel.avia.subscriptions.app.api.exceptions import (
    InvalidEmail, InvalidPromoSubscription, InvalidTravelVertical,
    InvalidUserAuthType, EmailNotFound, NoAccess
)
from travel.avia.subscriptions.app.api.schemas import (
    BaseNVUserSubscriptionPutRequestSchema,
    PutStatsSchema, UserSubscriptionGetResponseSchema,
    UserSubscriptionGetRequestQueryStringSchema,
    UserPromoSubscriptionDeleteRequestSchema
)
from travel.avia.subscriptions.app.model.storage import UpsertAction


class UserPromoSubscriptionPutRequestSchema(BaseNVUserSubscriptionPutRequestSchema):
    promo_subscription_code = fields.Str(example='travel_news')


class UserPromoSubscriptionDeleteResponseSchema(Schema):
    promo_subscription_code = fields.Str(example='travel_news')
    unsubscription_success = fields.Bool()


def get_user_promo_subscription_view(app: App):
    class UserPromoSubscriptionView(web.View):
        @docs(
            tags=['user_promo_subscription'],
            summary='Subscribes a user to a promo subscription',
            description='Subscribes a user to a promo subscription by his email, '
                        'credentials (session, passport, iCookie), and other meta parameters',
        )
        @request_schema(
            UserPromoSubscriptionPutRequestSchema,
        )
        @response_schema(
            PutStatsSchema,
            description='Number of created and updated subscriptions in a database.'
                        ' Depens on number of credentials passed.'
        )
        async def put(self):
            data = self.request['data']
            try:
                result = await app.user_promo_subscription_actor.put(**data)
                return json_response(
                    {'updated': len(result[UpsertAction.UPDATE]), 'created': len(result[UpsertAction.INSERT])}
                )
            except (InvalidPromoSubscription, InvalidTravelVertical, InvalidUserAuthType, InvalidEmail) as e:
                return HTTPBadRequest(text='{}'.format(e))

        @docs(
            tags=['user_promo_subscription'],
            summary='Get a list of user subscriptions',
            description='Get a list of user subscriptions, '
                        'identified by email and credentials (session, passport, iCookie).'
                        'Example: `/email=user@domain.com&auth_type=session&auth_value=123` will list all subscriptions'
                        'for user@domain.com that were created using unauthorized session "123".'
                        '\n'
                        'Email parameter is optional and is used as a filter.',
        )
        @querystring_schema(UserSubscriptionGetRequestQueryStringSchema)
        @response_schema(UserSubscriptionGetResponseSchema(many=True))
        async def get(self):
            data = {
                'email': self.request['querystring'].get('email'),
                'credentials': [
                    (
                        self.request['querystring']['auth_type'],
                        self.request['querystring']['auth_value'],
                    ),
                ],
            }
            try:
                result = await app.user_promo_subscription_actor.get_subscriptions_list(
                    **data, raise_on_auth_type_absent=True
                )
                return json_response(list(result))
            except (InvalidPromoSubscription, InvalidTravelVertical,
                    InvalidUserAuthType, InvalidEmail) as e:
                return HTTPBadRequest(text='{}'.format(e))

        @docs(
            tags=['user_promo_subscription'],
            summary='Delete subscriptions of user',
            description='Delete subscriptions of user by his email, '
                        'credentials (session, passport, iCookie), and other meta parameters',
        )
        @request_schema(UserPromoSubscriptionDeleteRequestSchema)
        @response_schema(UserPromoSubscriptionDeleteResponseSchema(many=True))
        async def delete(self):
            data = self.request['data']
            try:
                promo_subscription_names = await app.user_promo_subscription_actor.delete(**data)
                result = []

                for subscription in data['subscription_codes']:
                    result.append({
                        'promo_subscription_code': subscription,
                        'unsubscription_success': subscription in promo_subscription_names
                    })

                return json_response(result)
            except (NoAccess, EmailNotFound) as e:
                return HTTPBadRequest(text='{}'.format(e))

    return UserPromoSubscriptionView
