# coding=utf-8
from __future__ import unicode_literals

from aiohttp.web import json_response, HTTPBadRequest, View
from aiohttp_apispec import docs, request_schema, response_schema
from marshmallow import Schema, fields, validate

from travel.avia.subscriptions.app.api.app import App
from travel.avia.subscriptions.app.api.exceptions import (
    InvalidEmail, InvalidPromoSubscription, InvalidPriceChangeSubscription, InvalidTravelVertical,
    InvalidUserAuthType, EmailNotFound, NoAccess
)
from travel.avia.subscriptions.app.api.schemas import (
    BaseNVUserSubscriptionPutRequestSchema, CredentialsField, PutStatsSchema
)
from travel.avia.subscriptions.app.model.schemas import FilterSchema, MinPriceSchema
from travel.avia.subscriptions.app.model.storage import UpsertAction


class SubscriptionParams(Schema):
    filter_ = fields.Nested(FilterSchema, required=False, data_key='filter')
    date_range = fields.Integer(
        required=False,
        missing=1,
        validate=validate.Range(min=1, max=120),
    )
    min_price = fields.Nested(MinPriceSchema, required=False)


class SubscriptionSchema(Schema):
    subscription_type = fields.String(
        example='promo',
        required=True,
        validate=validate.Length(min=1, error='Subscription type can not be blank'),
    )
    subscription_code = fields.String(
        example='travel_news',
        required=True,
        validate=validate.Length(min=1, error='Subscription code can not be blank')
    )
    subscription_params = fields.Nested(SubscriptionParams, required=False)


class UserSubscriptionDeleteRequestSchema(Schema):
    subscriptions = fields.List(
        fields.Nested(
            SubscriptionSchema,
            exclude=['subscription_params'],
        ),
    )
    credentials = CredentialsField
    email = fields.Email()


class UnsubscriptionEventSchema(SubscriptionSchema):
    unsubscription_success = fields.Bool()


class UserSubscriptionDeleteResponseSchema(Schema):
    result = fields.List(fields.Nested(UnsubscriptionEventSchema))


class UserSubscriptionPutRequestSchema(BaseNVUserSubscriptionPutRequestSchema):
    subscriptions = fields.List(fields.Nested(SubscriptionSchema), required=True)


class UserSubscriptionPutResponseSchema(PutStatsSchema):
    subscription_type = fields.String(example='promo')


def get_user_subscription_view(app: App):
    class UserSubscriptionView(View):
        @docs(
            tags=['user_subscription'],
            summary='Subscribes a user to a subscription',
            description='Subscribes a user to a subscription by his email, '
                        'credentials (session, passport, iCookie), and other meta parameters',
        )
        @request_schema(
            UserSubscriptionPutRequestSchema,
        )
        @response_schema(
            UserSubscriptionPutResponseSchema,
            description='Number of created and updated subscriptions in a database.'
                        ' Depens on number of credentials passed.'
        )
        async def put(self):
            data = self.request['data']
            try:
                result = await app.user_subscription_actor.put(**data)
                return json_response([
                    {
                        'subscription_type': subscription_type,
                        'updated': len(changes[UpsertAction.UPDATE]),
                        'created': len(changes[UpsertAction.INSERT])
                    }
                    for subscription_type, changes in result.items()
                ])
            except (InvalidPromoSubscription, InvalidPriceChangeSubscription, InvalidTravelVertical,
                    InvalidUserAuthType, InvalidEmail) as e:
                return HTTPBadRequest(text='{}'.format(e))

        @docs(
            tags=['user_subscription'],
            summary='Delete subscriptions of user',
            description='Delete subscriptions of user by his email, '
                        'credentials (session, passport, iCookie), and other meta parameters',
        )
        @request_schema(UserSubscriptionDeleteRequestSchema)
        @response_schema(UserSubscriptionDeleteResponseSchema)
        async def delete(self):
            data = self.request['data']

            try:
                deleted_subscriptions = await app.user_subscription_actor.delete(**data)
                result = []

                for subscription in data['subscriptions']:
                    s_type, s_code = subscription['subscription_type'], subscription['subscription_code']

                    result.append({
                        'subscription_type': s_type,
                        'subscription_code': s_code,
                        'unsubscription_success': s_code in deleted_subscriptions.get(s_type, [])
                    })

                return json_response({
                    'result': result
                })
            except (NoAccess, EmailNotFound) as e:
                return HTTPBadRequest(text='{}'.format(e))

    return UserSubscriptionView
