from aiohttp.web_exceptions import HTTPBadRequest
from aiohttp.web_response import json_response
from aiohttp_apispec import docs, request_schema, response_schema
from marshmallow import Schema, fields

from travel.avia.subscriptions.app.api.app import App
from travel.avia.subscriptions.app.api.schemas import CredentialsField
from travel.avia.subscriptions.app.lib.qkey import QkeyValidationError


class UseSubscriptionSubscribedPostRequestSchema(Schema):
    credentials = CredentialsField
    email = fields.Email(example='user@domain.com', required=True)
    qkey = fields.Str(
        required=False,
        example='c213_c2_2017-12-21_2018-01-05_economy_1_0_1_ru'
    )
    qid = fields.Str(
        required=False,
        example=(
            '171206-144220-171.ticket.plane.'
            'c54_c239_2017-12-28_2018-01-04_economy_1_0_0_ru.'
            'ru'
        )
    )


class UseSubscriptionSubscribedPostResponseSchema(Schema):
    subscribed = fields.Bool()
    filter = fields.Str(allow_none=True, description='url postfix')
    qkey = fields.Str(allow_none=True, example='c213_c2_2017-12-21_2018-01-05_economy_1_0_1_ru')
    date_range = fields.Int(allow_none=True)


def get_user_subscription_subscribed_handler(app: App):
    @docs(
        tags=['user_subscription'],
        summary='Check subscribed user with email to the direction or not',
        description='Check subscribed user with email to the direction or not, '
                    'if there are many credentials handler return last updated subscription',
    )
    @request_schema(UseSubscriptionSubscribedPostRequestSchema)
    @response_schema(UseSubscriptionSubscribedPostResponseSchema)
    async def post(request):
        data = request['data']
        try:
            r = await app.user_price_change_subscription_actor.subscribed_on_direction(**data)
            return json_response(r)
        except QkeyValidationError as e:
            return HTTPBadRequest(text='{}'.format(e))

    return post
