import datetime
import logging
import secrets

from blackbox import BlackboxResponseError, XmlBlackbox
from enum import Enum
from typing import Optional, Union

from travel.avia.subscriptions.app.api.consts import PASSPORT_AUTH_TYPE
from travel.avia.subscriptions.app.api.exceptions import InvalidApprovingToken, UserAlreadyApproved
from travel.avia.subscriptions.app.api.interactor.user_token import UserTokenActor
from travel.avia.subscriptions.app.lib.sender import TransactionalApi as Sender
from travel.avia.subscriptions.app.model.db import User, Email, DBInstanceMixin
from travel.avia.subscriptions.app.settings.urls import subscriptions_cabinet_link, user_confirm_link

logger = logging.getLogger(__name__)


class OptinType(Enum):
    """
    Enum типа подтверждения подписки
    """
    SINGLE = 1
    DOUBLE = 2
    PENDING = 3


class UserConfirmActor:
    def __init__(self, user, email,
                 user_token_actor: UserTokenActor,
                 single_opt_in_sender: Sender,
                 double_opt_in_sender: Sender,
                 session_provider,
                 blackbox: XmlBlackbox,
                 force_confirm: bool = False,
                 ):
        self.User = user
        self.Email = email
        self.user_token_actor = user_token_actor
        self.single_opt_in_sender = single_opt_in_sender
        self.double_opt_in_sender = double_opt_in_sender
        self.session_provider = session_provider
        self.blackbox = blackbox
        self.force_confirm = force_confirm

    async def try_approve(
        self, session,
        user: User,
        email_obj: Union[Email, DBInstanceMixin],
        auth_type: str,
        auth_value: str,
        national_version: str,
    ):
        optin_type = await self._select_optin_type(
            user, email_obj.email, auth_type, auth_value
        )

        if optin_type is OptinType.SINGLE:
            await self._approve(session, user, email_obj, national_version)
        elif optin_type is OptinType.DOUBLE:
            user.approving_token = secrets.token_urlsafe(nbytes=32)
            await self._send_2opt_in(
                user.approving_token,
                email_obj.email,
                national_version
            )
            session.flush((user,))

    async def confirm(self, approving_token: str, national_version: str):
        with self.session_provider() as session:
            user: User = self.User(session).get(approving_token=approving_token)
            if user is None:
                raise InvalidApprovingToken('Token not found')

            if user.approved_at is not None:
                raise UserAlreadyApproved('User has been already approved')

            email_obj: Email = self.Email(session).get(id=user.email_id)
            await self._approve(session, user, email_obj, national_version)

    async def _select_optin_type(
        self,
        user: User,
        email: str,
        auth_type: str,
        auth_value: str,
    ) -> Optional[OptinType]:
        """
        Логика выбора между 1opt-in и 2opt-in для пользователя
        :return: Тип подтверждения 1opt-in или 2opt-in
        :rtype: OptinType
        """
        if self.force_confirm:
            logger.info('Automatically approving user %s with %s. Reason: force_user_confirm setting', email, auth_type)
            return OptinType.SINGLE

        try:
            if auth_type == PASSPORT_AUTH_TYPE:
                if email in self.blackbox.list_emails(auth_value):
                    return OptinType.SINGLE
        except BlackboxResponseError:
            logger.error('Cannot get emails list from blackbox')

        if user.approved_at is not None:
            return OptinType.SINGLE

        if user.approving_token is not None:
            return OptinType.PENDING

        return OptinType.DOUBLE

    async def _approve(
        self,
        session,
        user: User,
        email_obj: Email,
        national_version: str
    ):
        user.approved_at = datetime.datetime.utcnow()
        _, _, token = await self.user_token_actor.get_token_or_create(
            session=session, email_id=email_obj.id
        )
        await self._send_1opt_in(token, email_obj.email, national_version)
        session.flush((user,))

    async def _send_1opt_in(
        self,
        email_token: str,
        email: str,
        national_version: str
    ):
        if self.force_confirm:
            logger.info(
                'Not sending 1-opt-in for %s. Reason: Blocked by automatic confirmation force_user_confirm setting',
                email,
            )
            return

        args = {
            'cabinet_link': subscriptions_cabinet_link(email_token, national_version)
        }

        await self.single_opt_in_sender.send(email, async_send=False, args=args)

    async def _send_2opt_in(
        self,
        approving_token: str,
        email: str,
        national_version: str
    ):
        if self.force_confirm:
            logger.info(
                'Not sending 2-opt-in for %s. Reason: Blocked by automatic confirmation force_user_confirm setting',
                email,
            )
            return

        args = {
            'subscribe_link': user_confirm_link(approving_token, national_version)
        }

        await self.double_opt_in_sender.send(email, async_send=False, args=args)
