import logging

from blackbox import XmlBlackbox
from collections import defaultdict
from typing import Callable, ContextManager, List, Tuple, Dict, Iterable
from sqlalchemy.orm import Session

from travel.avia.subscriptions.app.api.exceptions import InvalidPromoSubscription, UnknownPointsError
from travel.avia.subscriptions.app.api.util_db import authenticate, email_get_or_fail, travel_vertical_get_or_fail
from travel.avia.subscriptions.app.api.interactor.user_price_change_subscription import (
    UserPriceChangeSubscriptionActor
)
from travel.avia.subscriptions.app.api.interactor.user_promo_subscription import (
    UserPromoSubscriptionActor
)
from travel.avia.subscriptions.app.model.storage import Storage, UpsertAction

logger = logging.getLogger(__name__)


class UserSubscriptionActor:
    def __init__(
        self,
        email: Callable[[Session], Storage],
        user_auth_type: Callable[[Session], Storage],
        user_auth: Callable[[Session], Storage],
        user: Callable[[Session], Storage],
        user_promo_subscription_actor: UserPromoSubscriptionActor,
        user_price_change_subscription_actor: UserPriceChangeSubscriptionActor,
        travel_vertical: Callable[[Session], Storage],
        session_provider: Callable[[], ContextManager[Session]],
        blackbox: XmlBlackbox
    ):
        self.Email = email
        self.UserAuthType = user_auth_type
        self.UserAuth = user_auth
        self.User = user
        self.TravelVertical = travel_vertical
        self.session_provider = session_provider
        self.blackbox = blackbox
        self._user_promo_subscription_actor = user_promo_subscription_actor
        self._actor_by_type = {
            'promo': (user_promo_subscription_actor, self._promo_actor_args_provider),
            'price': (user_price_change_subscription_actor, self._price_actor_args_provider),
        }

    async def delete(
        self, *,
        subscriptions: List[Dict],
        credentials: List[Tuple[str, str]],
        email: str
    ) -> Dict[str, Iterable[str]]:
        result = defaultdict(str)
        with self.session_provider() as session:
            email_obj = email_get_or_fail(self.Email(session), email)
            authenticate(
                credentials=credentials,
                email_obj=email_obj,
                user_dbs=self.User(session),
                user_auth_dbs=self.UserAuth(session),
                user_auth_type_dbs=self.UserAuthType(session),
                blackbox=self.blackbox
            )
            groups = self._group_by_subscription_type(subscriptions)

            for subscription_type, subscriptions in groups.items():
                actor, _ = self._actor_by_type.get(subscription_type, (None, None))
                if actor is None:
                    continue
                codes = [s['subscription_code'] for s in subscriptions]
                deleted_subscriptions = await actor.delete(
                    session=session,
                    subscription_codes=codes,
                    credentials=credentials,
                    email=email_obj,
                    authenticated=True,
                )
                result[subscription_type] = deleted_subscriptions

            return result

    async def put(
        self, *,
        source: str,
        language: str,
        national_version: str,
        subscriptions: Iterable[Dict[str, str]],
        timezone: str,
        credentials: Iterable[Tuple[str, str]],
        name: str,
        email: str,
        travel_vertical_name: str,
    ):
        kwargs = dict(
            email=email,
            name=name,
            credentials=credentials,
            source=source,
            national_version=national_version,
            language=language,
            timezone=timezone,
            travel_vertical_name=travel_vertical_name
        )
        groups = self._group_by_subscription_type(subscriptions)
        result = {
            subscription_type: {
                UpsertAction.INSERT: [],
                UpsertAction.UPDATE: []
            }
            for subscription_type in groups
        }

        with self.session_provider() as session:
            kwargs['email'] = self.Email(session).get_or_create(email=email)
            kwargs['travel_vertical'] = travel_vertical_get_or_fail(
                travel_vertical_dbs=self.TravelVertical(session),
                travel_vertical_name=travel_vertical_name
            )
            # Хотим, чтобы все подписки клали в рамках одной сессии
            kwargs['session'] = session

            for subscription_type, subscriptions in groups.items():
                actor, arg_provider = self._actor_by_type.get(subscription_type, (None, None))
                if actor is None:
                    continue

                for subscription in subscriptions:
                    try:
                        put_result = await actor.put(**arg_provider(subscription, **kwargs))
                        for action, result_subscriptions in put_result.items():
                            result[subscription_type][action].extend(result_subscriptions)
                    except (InvalidPromoSubscription, UnknownPointsError) as e:
                        # ошибки, которые нужно зафиксировать и проигнорировать
                        logger.exception(
                            f'Skip subscription({subscription["subscription_code"]}): {str(e)}'
                        )

        return result

    @staticmethod
    def _group_by_subscription_type(subscriptions):
        groups = defaultdict(list)

        for subscription in subscriptions:
            groups[subscription['subscription_type']].append(subscription)

        return groups

    @staticmethod
    def _price_actor_args_provider(subscription: dict, **kwargs):
        args = {
            'qkey': subscription['subscription_code'],
            **(subscription.get('subscription_params') or {}),
            **kwargs
        }
        args.pop('national_version')

        return args

    @staticmethod
    def _promo_actor_args_provider(subscription: dict, **kwargs):
        return {
            'promo_subscription_code': subscription['subscription_code'],
            **kwargs
        }
