import asyncio
import datetime
import secrets

from sqlalchemy import distinct

import travel.avia.subscriptions.app.model.db as db_models

from travel.avia.subscriptions.app.api.consts import TOKEN_AUTH_TYPE
from travel.avia.subscriptions.app.api.util_db import user_auth_type_get_or_fail


class UserTokenActor:
    def __init__(
        self,
        email,
        user_auth_type,
        user_auth,
        user_promo_subscription,
        user,
        session_provider,
    ):
        self.Email = email
        self.UserAuthType = user_auth_type
        self.UserAuth = user_auth
        self.UserPromoSubscription = user_promo_subscription
        self.User = user
        self.session_provider = session_provider

    async def get(self):
        with self.session_provider() as session:
            auth_type = self._get_token_auth_type(session)

            token_tuples = await asyncio.gather(
                *(
                    self.get_token_or_create(session, email_id_tuple[0], auth_type)
                    for email_id_tuple in await self._get_all_approved_email_ids(session)
                )
            )

            for token_tuple in token_tuples:
                yield token_tuple

    async def get_token_or_create(self, session, email_id, token_auth_type=None):
        if token_auth_type is None:
            token_auth_type = self._get_token_auth_type(session)

        email_obj = self.Email(session).get(id=email_id)
        name, token = await self._get_user_with_token(session, email_obj, token_auth_type)

        if token is None:
            user_auth = self.UserAuth(session).create(
                user_auth_type_id=token_auth_type.id,
                auth_value=secrets.token_urlsafe(nbytes=32)
            )
            self.User(session).create(
                user_auth_id=user_auth.id,
                name=name,
                email_id=email_obj.id,
                approved_at=datetime.datetime.utcnow()
            )
            token = user_auth.auth_value

        return email_obj.email, name, token

    def _get_token_auth_type(self, session) -> db_models.UserAuthType:
        return user_auth_type_get_or_fail(
            self.UserAuthType(session), TOKEN_AUTH_TYPE
        )

    @staticmethod
    async def _get_all_approved_email_ids(session):
        return (session.query(distinct(db_models.User.email_id))
                .join(db_models.UserPromoSubscription,
                      db_models.UserPromoSubscription.user_id == db_models.User.id)
                .filter(db_models.UserPromoSubscription.deleted_at.is_(None))
                .filter(db_models.User.deleted_at.is_(None))
                .filter(db_models.User.approved_at.isnot(None))
                .all())

    async def _get_user_with_token(self, session, email_obj, token_auth_type):
        user = None

        for user in self.User(session).find(email_id=email_obj.id):
            user_auth = self.UserAuth(session).get(
                id=user.user_auth_id,
                user_auth_type_id=token_auth_type.id
            )

            if user_auth is not None:
                return user.name, user_auth.auth_value

        # Не может быть None, так как хотя бы одна
        # запись из user_auth соотвествует данному email
        return user.name, None
