from marshmallow import Schema, fields, validate

ALLOWED_AUTH_TYPES = (
    'session',
    'iCookie',
    'passportId',
    'yandexUid',
    'token',
)

ALLOWED_LANGUAGES = (
    'en',
    'ru',
    'uk',
    'tr',
    'de',
)

ALLOWED_NATIONAL_VERSIONS = (
    'com',
    'ru',
    'ua',
    'tr',
    'kz',
)

CredentialsField = fields.List(
    fields.Tuple(
        (
            fields.String(
                required=True,
                # TODO(mikhailche): I don't like this solution.
                #  It blurs the area of responsibility for validating auth type
                #  between database-driven validator later in code and hardcoded one.
                #  Consider filling allowed_auth_types from user_auth_type table or
                #  removing user_auth_type table
                validate=validate.OneOf(ALLOWED_AUTH_TYPES),
            ),
            fields.String(
                required=True,
                validate=validate.Length(min=1, error='Auth value should not be blank')
            ),
        )
    ),
    example=(
        ('session', '123'),
        ('iCookie', '456'),
    ),
    required=True
)


class PutStatsSchema(Schema):
    updated = fields.Int(required=True)
    created = fields.Int(required=True)


class BaseUserSubscriptionPutRequestSchema(Schema):
    email = fields.Email(example='user@domain.com', required=True)
    name = fields.String(example='Vasily', required=True)
    credentials = CredentialsField
    source = fields.String(
        example='wizard',
        required=True,
        validate=validate.Length(min=1, error='Source field should not be blank')
    )
    language = fields.String(
        example='en',
        required=True,
        validate=validate.OneOf(ALLOWED_LANGUAGES),
    )
    timezone = fields.String(example='Asia/Yekaterinburg', required=True)
    travel_vertical_name = fields.String(
        example='trains',
        required=True,
        validate=validate.Length(min=1, error='Travel vertical name should not be blank')
    )


class BaseNVUserSubscriptionPutRequestSchema(BaseUserSubscriptionPutRequestSchema):
    national_version = fields.String(
        example='com',
        required=True,
        validate=validate.OneOf(ALLOWED_NATIONAL_VERSIONS)
    )


class UserSubscriptionGetResponseSchema(Schema):
    email = fields.Email(example='user@domain.com')
    subscription_code = fields.String(example='travel_news')
    national_version = fields.String(example='com')
    language = fields.String(example='en')
    source = fields.String(example='wizard')
    name = fields.String(example='Подписка на новости Яндекс.Путешествий')
    url = fields.String()


class UserSubscriptionGetRequestQueryStringSchema(Schema):
    email = fields.Email(example='user@domain.com')
    auth_type = fields.String(
        required=True,
        example='session',
        validate=validate.OneOf(ALLOWED_AUTH_TYPES),
    )
    auth_value = fields.String(
        required=True,
        example='123',
        validate=validate.Length(min=1, error='Auth value should not be blank')
    )


class UserPromoSubscriptionDeleteRequestSchema(Schema):
    subscription_codes = fields.List(fields.String(example='travel_news'))
    credentials = CredentialsField
    email = fields.Email()
