import logging
import warnings
from functools import wraps

import travel.avia.subscriptions.app.api.app as api
import travel.avia.subscriptions.app.settings.app as app_settings
import ujson
from aiohttp import web
from aiohttp.web_exceptions import HTTPException
from aiohttp.web_response import json_response
from aiohttp_apispec import setup_aiohttp_apispec, validation_middleware
from travel.avia.subscriptions.app.api.handler.user_confirm import (
    get_user_confirm_handler
)
from travel.avia.subscriptions.app.api.handler.user_price_change_subscription import (
    get_user_price_change_subscription_view
)
from travel.avia.subscriptions.app.api.handler.user_promo_subscription import (
    get_user_promo_subscription_view
)
from travel.avia.subscriptions.app.api.handler.user_subscription import (
    get_user_subscription_view
)
from travel.avia.subscriptions.app.api.handler.user_subscription_list import (
    get_user_subscription_list_handler
)
from travel.avia.subscriptions.app.api.handler.user_subscription_subscribed import (
    get_user_subscription_subscribed_handler
)
from travel.avia.subscriptions.app.api.handler.user_token import (
    get_user_token_handler
)
from travel.avia.subscriptions.app.lib.pingtools import PingResponse

logger = logging.getLogger(__name__)


def json_view(func):
    @wraps(func)
    async def wrapper(request: web.Request, *args, **kwargs):
        logger.info('Request {}, params: {} {}'.format(request, args, kwargs))
        http_status_code = 200
        response = ''
        with warnings.catch_warnings(record=True) as warnings_list:
            # Cause all warnings to always be triggered.
            warnings.simplefilter("always")
            try:
                request_dict = {}
                if request.can_read_body:
                    request_dict = await request.json()
                if isinstance(request_dict, str):
                    raise ValueError("Invalid json: {}".format(request_dict))
                if isinstance(request_dict, dict):
                    response = await func(**request_dict)
                else:
                    response = await func(*request_dict)
            except HTTPException as http_exc:
                return json_response(
                    data=http_exc.text or http_exc.body,
                    status=http_exc.status,
                    reason=http_exc.reason,
                    dumps=ujson.dumps,
                )
            except Exception as exc:
                logger.exception('Exception occured in function {}. Returning json error')
                response = repr(exc)
                http_status_code = 500
            if warnings_list:
                logger.warning('{}'.format(warnings_list))

        return json_response(
            data=response,
            status=http_status_code,
            dumps=ujson.dumps,
        )

    return wrapper


class WebApp:
    def __init__(self, config: app_settings.AppConfig):
        self.config: app_settings.AppConfig = config
        self.app = api.App(
            config=self.config,
        )

        self.web_app = web.Application(logger=logger)
        self.web_app.on_startup.append(self.load_dicts)

        self.web_app.router.add_route(
            'GET', '/ping',
            json_view(self.ping)
        )
        self.web_app.router.add_route(
            'GET', '/user-confirm',
            get_user_confirm_handler(self.app)
        )
        self.web_app.router.add_route(
            'GET', '/user-token',
            get_user_token_handler(self.app)
        )
        self.web_app.router.add_route(
            'POST', '/user-subscription/list',
            get_user_subscription_list_handler(self.app)
        )
        self.web_app.router.add_route(
            'POST', '/user-subscription/subscribed',
            get_user_subscription_subscribed_handler(self.app)
        )
        self.web_app.router.add_view(
            '/user-promo-subscription',
            get_user_promo_subscription_view(self.app)
        )
        self.web_app.router.add_view(
            '/user-price-change-subscription',
            get_user_price_change_subscription_view(self.app)
        )
        self.web_app.router.add_view(
            '/user-subscription',
            get_user_subscription_view(self.app)
        )

        self.web_app.middlewares.append(validation_middleware)
        setup_aiohttp_apispec(
            app=self.web_app,
            title="Subscriptions",
            version="v1",
            swagger_path='/api/docs',
        )

    async def load_dicts(self, web_app):
        logger.info('Async loading dicts on startup')
        await self.app.point_key_resolver.prepare_dicts_async()

    async def ping(self):
        result: PingResponse = await self.app.ping()
        if not result.ready:
            raise web.HTTPServiceUnavailable(reason=ujson.dumps(result))
        return result

    def run(self):
        web.run_app(
            self.web_app,
            host=self.config.server.host,
            port=self.config.server.port,
        )
