import asyncio
import logging
from datetime import datetime, timedelta
from threading import Thread
from typing import ClassVar, Optional, Tuple

from travel.avia.subscriptions.app.lib.pingtools import PingResponse

import aiohttp
from sandbox.common.rest import Client
from travel.library.python.dicts.base_repository import BaseRepository
from travel.library.python.dicts.country_repository import CountryRepository
from travel.library.python.dicts.region_repository import RegionRepository
from travel.library.python.dicts.settlement_repository import SettlementRepository
from travel.library.python.dicts.station_repository import StationRepository

logger = logging.getLogger(__name__)


class ResourceNotFound(Exception):
    pass


class ResourceHasNoHttpLinks(Exception):
    pass


def async_to_sync(fn):
    def runme():
        loop = asyncio.new_event_loop()
        loop.run_until_complete(fn)
        loop.close()

    thread = Thread(target=runme)
    thread.start()
    thread.join()


class Dict:
    def __init__(
        self,
        repository_cls: ClassVar[BaseRepository],
        resource_type: str,
        check_update_interval: Optional[timedelta] = None,
        oauth_token: Optional[str] = None,
    ):
        self._repository_cls = repository_cls
        self._repository: BaseRepository = self._repository_cls()
        self._resource_type = resource_type
        self._sandbox = Client(auth=oauth_token)
        self._last_resource_hash = None
        self._resource_updated_event = asyncio.Event()
        self._last_update_time = None
        self._check_update_interval = check_update_interval
        if check_update_interval is None:
            self._check_update_interval = timedelta(days=1, minutes=5)
        # Сбросим событие, так как мы сейчас не обновляемся
        self._resource_updated_event.set()

    async def get(self, id_: int):
        await self.prepare_repositry()

        # Если начали обновлять справочники, то
        # ждем пока не обновятся
        await self._resource_updated_event.wait()
        return self._repository.get(id_)

    async def prepare_repositry(self, in_background=True):
        if self._may_update():
            # Защита от дурака, хотим, чтобы обновление было одно за раз
            assert self._resource_updated_event.is_set(), 'Dicts are already updating'
            # Инициируем ожидание обновления справочников
            self._resource_updated_event.clear()
            if in_background:
                asyncio.create_task(self._prepare_repository())
            else:
                await self._prepare_repository()

    async def _prepare_repository(self):
        try:
            resource_info, fresh = await self._updated_resource()
            if fresh:
                return

            content = await self._download_dict(resource_info)
            logger.info(f'Resource {resource_info["type"]} has been downloaded')
            # Обнуляем справочники в памяти, так как пришли новые
            self._repository: BaseRepository = self._repository_cls()
            self._repository.load_from_string(content)

            # Фиксируем время последнего обновления и хеш
            # только после того, как получилось загрузить новый
            # справочник в память
            self._last_update_time = datetime.now()
            self._last_resource_hash = resource_info['md5']
        except Exception as e:
            logger.exception(e)
        finally:
            # Заканчиваем ожидание обновления справочников
            self._resource_updated_event.set()

    async def _updated_resource(self) -> Tuple[Optional[dict], bool]:
        infos = self._sandbox.resource.read(
            limit=1,
            type=self._resource_type,
            order='-created',
            state='READY',
        )

        if len(infos['items']) < 1:
            raise ResourceNotFound(f'Resource({self._resource_type}) not found in sandbox')

        resource_info = infos['items'][0]
        # Если содержимое не поменялось, то не будем загружать
        # в память новые справочники
        if self._last_resource_hash == resource_info['md5']:
            return None, True

        return resource_info, False

    def _may_update(self):
        # уже обновляем, повторно не нужно делать запрос
        if not self._resource_updated_event.is_set():
            return False
        # никогда до этого не обновляли, поэтому нужно обновить справочники
        if self._last_update_time is None:
            return True
        # предполагаем, что обновления будут раз в определенный период
        if datetime.now() - self._last_update_time >= self._check_update_interval:
            return True

        return False

    @staticmethod
    async def _download_dict(resource_info: dict) -> bytes:
        urls = [resource_info['http']['proxy']] + resource_info['http']['links']
        if len(urls) == 0:
            logger.info('No urls for resource: %s', resource_info)
            raise ResourceHasNoHttpLinks(f'Resource {resource_info["type"]} has no http links')
        async with aiohttp.ClientSession() as session:
            for url in urls:
                async with session.get(url) as resp:
                    if resp.status != 200:
                        logger.warning(f'Couldn\'t download resource from {url}')
                        continue
                    return await resp.read()

    async def ping(self) -> PingResponse:
        return PingResponse(
            self._last_update_time is not None,
            {
                'last_updated': self._last_update_time,
                'updating': not self._resource_updated_event.is_set(),
            },
        )


class PointKeyResolver:
    def __init__(self, oauth_token: Optional[str] = None):
        self._station_repository = Dict(
            repository_cls=StationRepository,
            resource_type='TRAVEL_DICT_RASP_STATION_PROD',
            oauth_token=oauth_token,
        )
        self._settlement_repository = Dict(
            repository_cls=SettlementRepository,
            resource_type='TRAVEL_DICT_RASP_SETTLEMENT_PROD',
            oauth_token=oauth_token,
        )
        self._region_repository = Dict(
            repository_cls=RegionRepository,
            resource_type='TRAVEL_DICT_RASP_REGION_PROD',
            oauth_token=oauth_token,
        )
        self._country_repository = Dict(
            repository_cls=CountryRepository,
            resource_type='TRAVEL_DICT_RASP_COUNTRY_PROD',
            oauth_token=oauth_token,
        )

    def prepare_dicts(self):
        async_to_sync(self.prepare_dicts_async())

    async def prepare_dicts_async(self):
        await self._station_repository.prepare_repositry()
        await self._settlement_repository.prepare_repositry()
        await self._region_repository.prepare_repositry()
        await self._country_repository.prepare_repositry()

    async def resolve(self, point_key: str) -> Optional:
        try:
            prefix, pk = point_key[0], point_key[1:]
            pk = int(pk)

        except (IndexError, TypeError):
            raise ValueError("Unknown key %r" % point_key)

        if prefix == 'c':
            return await self._settlement_repository.get(pk)
        elif prefix == 's':
            return await self._station_repository.get(pk)
        elif prefix == 'r':
            return await self._region_repository.get(pk)
        elif prefix == 'l':
            return await self._country_repository.get(pk)

        raise ValueError("Unknown type prefix %s" % prefix)

    async def ping(self) -> PingResponse:
        info = {
            'station': await self._station_repository.ping(),
            'settlement': await self._settlement_repository.ping(),
            'region': await self._region_repository.ping(),
            'country': await self._country_repository.ping(),
        }
        return PingResponse(
            info['station'].ready and info['settlement'].ready and info['region'].ready and info['country'].ready, info
        )
