# -*- coding: utf-8 -*-
from dataclasses import dataclass
from datetime import datetime, date
from typing import Optional

QKEY_DATE_FORMAT = '%Y-%m-%d'


@dataclass
class QkeyStructure:
    point_from_key: str
    point_to_key: str
    date_forward: datetime
    date_backward: Optional[datetime]
    klass: str
    adults: int
    children: int
    infants: int
    national_version: str


class QkeyValidationError(Exception):
    pass


def qkey_from_params(
    point_from_key=None, point_to_key=None,
    date_forward=None, date_backward=None,
    klass=None, adults=1, children=0, infants=0,
    national_version=None
):
    """
    :param str point_from_key: example: "c213", "s65646"
    :param str point_to_key: example: "c213", "s65646"
    :param datetime | date | str date_forward: example: "2019-06-30", date(2019,6,30), datetime(2019,6,30)
    :param None |datetime | date | str date_backward: same format as date_forward
    :param str klass: example: "economy", "business"
    :param str | int adults:
    :param str | int children:
    :param str | int infants:
    :param str national_version: example: "ru"
    :rtype: str
    """
    if not point_from_key:
        raise ValueError('Empty point_from_key')
    if not point_to_key:
        raise ValueError('Empty point_to_key')
    if not date_forward:
        raise ValueError('Empty date_forward')
    if not klass:
        raise ValueError('Empty klass')
    if not national_version:
        raise ValueError('Empty nv')

    if isinstance(date_forward, (datetime, date)):
        date_forward = date_forward.strftime(QKEY_DATE_FORMAT)
    elif isinstance(date_forward, str):
        date_forward_reformat = datetime.strptime(date_forward, QKEY_DATE_FORMAT)
        date_forward = date_forward_reformat.strftime(QKEY_DATE_FORMAT)
    else:
        raise ValueError('Invalid date_forward. Value %s, type %s' % (date_forward, type(date_forward)))

    if isinstance(date_backward, (datetime, date)):
        date_backward = date_backward.strftime(QKEY_DATE_FORMAT)
    elif isinstance(date_backward, str):
        date_backward_reformat = datetime.strptime(date_backward, QKEY_DATE_FORMAT)
        date_backward = date_backward_reformat.strftime(QKEY_DATE_FORMAT)
    elif date_backward is None:
        date_backward = str(date_backward)
    else:
        raise ValueError('Invalid date_backward. Value %s, type %s' % (date_backward, type(date_backward)))

    if isinstance(adults, int):
        adults = str(adults)
    if isinstance(children, int):
        children = str(children)
    if isinstance(infants, int):
        infants = str(infants)
    values = [
        point_from_key, point_to_key, date_forward,
        date_backward, klass, adults, children, infants,
        national_version
    ]
    type_ok = all(map(lambda v: isinstance(v, str), values))
    if not type_ok:
        raise ValueError('Incorrect input value type')
    return '_'.join(values)


def structure_from_qkey(qkey: str):
    """
    :param str qkey:
    :rtype: QkeyStructure
    """
    point_from, point_to, date_forward, date_backward, klass, adults, children, infants, nv = qkey.split('_')
    date_forward = datetime.strptime(date_forward, QKEY_DATE_FORMAT)
    date_backward = datetime.strptime(date_backward, QKEY_DATE_FORMAT) if date_backward != 'None' else None

    adults = int(adults)
    children = int(children)
    infants = int(infants)

    if adults + children + infants < 1:
        raise ValueError('Incorrect total number of passengers')

    return QkeyStructure(
        point_from_key=point_from,
        point_to_key=point_to,
        date_forward=date_forward,
        date_backward=date_backward,
        klass=klass,
        adults=adults,
        children=children,
        infants=infants,
        national_version=nv,
    )


def qkey_from_qid(qid: str):
    """
    :param str qid:
    :rtype: str
    """
    name_parts = qid.split('.')
    return name_parts[3]


def validate_qkey(qid: Optional[str], qkey: Optional[str]):
    """
    :param str qid:
    :param str qkey:
    :rtype: str
    """
    if not qkey:
        if qid:
            qkey = qkey_from_qid(qid)
        else:
            raise QkeyValidationError('qid and qkey are not specified')
    else:
        if '.' in qkey or '$' in qkey:
            raise QkeyValidationError('qkey is invalid (should not contain "." or "$")')
    return qkey
