import json
import logging
import urllib.parse

from aiohttp import BasicAuth, ClientResponse  # noqa
from aiohttp_retry import RetryClient, ExponentialRetry

log = logging.getLogger(__name__)


class SenderError(Exception):
    pass


class TransactionalApi:
    def __init__(self, settings, campaign_slug: str):
        self.settings = settings
        self.campaign_slug = campaign_slug
        self.url = '{host}/api/0/{account_slug}/transactional/{campaign_slug}/'.format(
            host=self.settings.host, account_slug=self.settings.account_slug,
            campaign_slug=self.campaign_slug
        )
        self.auth = settings.auth

        # Wait fixed 2s
        self.retry_options = ExponentialRetry(
            attempts=3,
            factor=1,
            start_timeout=2,
            statuses=range(400, 600),
        )

    async def send(self, to_email, args, async_send=True, from_name=None, from_email=None,
                   countdown=None, expires=None, attachments=None):
        """

        NOTE: from_email, from_name - (это нужно в редких кейсах; обычно лучше использовать адрес,
                прописанный в шаблоне Рассылятора)
        :param basestring to_email: куда отправлять. См. также пункт "Отправка на uid"
        :param dict args: json с переменными, которые будут подставлены в тело письма
        :param bool async_send: отправить письмо сразу (синхронно, без очереди) или положить в очередь
        :param from_name: заменить имя отправителя, прописанное в шаблоне
        :param from_email: заменить адрес отправителя, прописанный в шаблоне
        :param basestring|int countdown: для асинхронной отправки - отправить рассылку не сразу,
                а через countdown секунд.
        :param basestring|int expires: для асинхронной отправки - не отправлять через expires секунд.
                через какое время уже не нужно отправлять письмо.
        :param attachments: приаттачить эти файлы к письму
        :return: response of sender
        :rtype: ClientResponse
        :raise: SenderError
        """
        if not self.campaign_slug:
            log.warning('Not sending email, because campaign slug is not configured')

        data = self.build_data(
            to_email, args, async_send,
            from_name, from_email, countdown,
            expires, attachments
        )

        url = urllib.parse.urljoin(self.url, 'send')

        async with RetryClient() as client:
            async with client.post(
                url, auth=BasicAuth(*self.auth), data=data, verify_ssl=False, retry_options=self.retry_options,
            ) as response:  # type: ClientResponse
                pass

            if 400 <= response.status < 599:
                raise SenderError(
                    'Send %s failed %r. Sender response %s, [%s]' % (
                        url, data,
                        response.status,
                        await response.read(),
                    )
                )
            return response

    @staticmethod
    def _to_sec_str(x):
        return '{}s'.format(x)

    def build_data(self, to_email, args, async_send, from_name, from_email,
                   countdown, expires, attachments):
        data = {
            'to_email': to_email,
            'args': json.dumps(args),
            'async': async_send,
        }

        if from_name is not None:
            data['from_name'] = from_name

        if from_email is not None:
            data['from_email'] = from_email

        if countdown is not None:
            data['countdown'] = self._to_sec_str(countdown)

        if expires is not None:
            data['expires'] = self._to_sec_str(expires)

        if attachments is not None:
            data['attachments'] = attachments

        return data
