# -*- coding: utf-8 -*-
import logging
import time
from enum import Enum
from typing import Optional, Tuple

import ujson

from travel.avia.subscriptions.app.lib.yt_loggers.yt_log import validate_schema

user_action_yt_logger = logging.getLogger('yt.subscriptions.user_action')
log = logging.getLogger(__name__)


class UserActionType(Enum):
    SUBSCRIBE = 'subscribe'
    UNSUBSCRIBE = 'unsubscribe'
    UNSUBSCRIBE_DIRECTION = 'unsubscribe_direction'
    CONFIRM = 'confirm'


class SubscriptionType(Enum):
    PROMO = 'promo'
    PRICE_CHANGE = 'price_change'


class NationalVersion(Enum):
    RU = 'ru'
    UA = 'ua'
    TR = 'tr'
    COM = 'com'
    KZ = 'kz'


class Language(Enum):
    RU = 'ru'
    EN = 'en'
    UK = 'uk'
    TR = 'tr'
    DE = 'de'


user_action_log_schema = dict(
    unixtime=dict(type=int, lf_type='VT_UINT64', required=True),
    action_type=dict(type=UserActionType, lf_type='VT_STRING', required=True),
    subscription_type=dict(type=SubscriptionType, lf_type='VT_STRING', required=True),
    code=dict(type=str, lf_type='VT_STRING', required=True),
    national_version=dict(type=NationalVersion, lf_type='VT_STRING', required=True),
    language=dict(type=Language, lf_type='VT_STRING', required=True),
    email=dict(type=str, lf_type='VT_STRING', required=True),
    date_range=dict(type=int, lf_type='VT_INT64', required=False),
    filter_params=dict(type=dict, lf_type='VT_ANY', required=False),
    travel_vertical=dict(type=str, lf_type='VT_STRING', required=False),
    source=dict(type=str, lf_type='VT_STRING', required=False),
    pending_passport=dict(type=str, lf_type='VT_STRING', required=False),
    pending_session=dict(type=str, lf_type='VT_STRING', required=False),
    min_price=dict(type=dict, lf_type='VT_ANY', required=False),
)


async def log_user_action(**kwargs):
    """
    :param kwargs: logging data according to user_action_log_schema
    """
    try:
        kwargs.update(dict(unixtime=int(time.time())))
        await validate_schema(kwargs, user_action_log_schema)
        user_action_yt_logger.info('%s', ujson.dumps(kwargs))
    except (KeyError, ValueError, TypeError) as e:
        log.exception('Cannot log data %s. %s', kwargs, e)


async def log_user_subscribed(
    *,
    subscription_type: SubscriptionType,
    code: str,
    national_version: NationalVersion,
    language: Language,
    email: str,
    date_range: Optional[int],
    filter_params: Optional[dict],
    travel_vertical: str,
    source: str,
    pending_passport: Optional[str],
    pending_session: Optional[str],
    min_price: Optional[Tuple[Optional[int], Optional[str]]],
):
    """
    :param subscription_type: defines if it's a promo, price_change or other type of Subscription
    :param code: subscription identifier. For promo it's something like "travel_news",
        for price_change it's a query identifier like "c213_c5_2019-10-15_2019-10-16_economy_1_0_0_ru"
    :param national_version: see NationalVersion enum
    :param language: see Language enum
    :param email: user's email
    :param date_range: for price change subscriptions - range (in days) of subscriptions
    :param filter_params: for price change subscriptions - filters like baggae filter,
        time of day filter, etc...
    :param travel_vertical: name of travel vertical (might be avia, trains, hotels, etc...)
    :param source: source of subscription (might be front, wizard, mobile or any other source type), not strict
    :param pending_passport: determines if any passport is waiting to be approved
    :param pending_session: determines if any session is waiting to be approved
    :param min_price: for price change subscriptions - what  was the last seen min price for the direction
    """
    await log_user_action(
        action_type=UserActionType.SUBSCRIBE,
        subscription_type=subscription_type,
        code=code,
        national_version=national_version,
        language=language,
        email=email,
        date_range=date_range,
        filter_params=filter_params,
        travel_vertical=travel_vertical,
        source=source,
        pending_passport=pending_passport,
        pending_session=pending_session,
        min_price=min_price,
    )
