import json

from marshmallow import Schema
from sqlalchemy import VARCHAR
from sqlalchemy.dialects.postgresql import JSONB, dialect as pg_dialect
from sqlalchemy.ext.mutable import Mutable
from sqlalchemy.types import TypeDecorator

from typing import Iterable, List, Optional, Type, Union


class MutableList(Mutable, list):
    @classmethod
    def coerce(cls, key, value):
        if not isinstance(value, MutableList):
            if isinstance(value, list):
                return MutableList(value)

            # вызов исключения ValueError
            return Mutable.coerce(key, value)
        else:
            return value

    def append(self, object_) -> None:
        super(MutableList, self).append(object_)
        self.changed()

    def extend(self, iterable: Iterable) -> None:
        super(MutableList, self).extend(iterable)
        self.changed()

    def insert(self, index: int, object_) -> None:
        super(MutableList, self).insert(index, object_)
        self.changed()

    def remove(self, object_) -> None:
        super(MutableList, self).remove(object_)
        self.changed()

    def pop(self, object_=-1):
        super(MutableList, self).pop(object_)
        self.changed()

    def __setitem__(self, key, value):
        super(MutableList, self).__setitem__(key, value)
        self.changed()

    def __delitem__(self, key):
        super(MutableList, self).__delitem__(key)
        self.changed()


class JSONEncodedDict(TypeDecorator):
    impl = VARCHAR

    def process_bind_param(self, value, dialect):
        if value is not None:
            value = json.dumps(value)

        return value

    def process_result_value(self, value, dialect):
        if value is not None:
            value = json.loads(value)
        return value


class Json(TypeDecorator):
    impl = JSONB

    def __new__(cls, *args, **kwargs):
        schema = kwargs.get('schema')
        instance = super(Json, cls).__new__(cls)

        if schema and not callable(schema) and schema.many:
            return MutableList.as_mutable(instance)

        return instance

    def __init__(self,
                 dataclass: Type,
                 schema: Union[Schema, Type[Schema]],
                 *args, **kwargs):
        super(Json, self).__init__(*args, **kwargs)
        self.dataclass = dataclass
        self.schema = None
        self.schema = schema() if callable(schema) else schema

    def process_bind_param(self, value, dialect) -> Optional[Union[dict, List[dict]]]:
        if value is not None:
            if self.schema.many and len(value) == 0:
                return []

            check_instance = value[0] if self.schema.many else value

            if not isinstance(check_instance, self.dataclass):
                raise ValueError(f'Value must be {self.dataclass}')

            return self.schema.dump(value)

        return value

    def process_result_value(self, value: Union[dict, List[dict]], dialect):
        if value is not None:
            if self.schema.many and len(value) == 0:
                return []

            loaded = self.schema.load(value)
            check_instance = loaded[0] if self.schema.many else loaded
            if isinstance(check_instance, self.dataclass):
                return loaded
            if self.schema.many:
                return [self.dataclass(**kwargs) for kwargs in loaded]
            return self.dataclass(**loaded)

        return value

    # Уточнение: Метод необходим для тестирования, так как
    # в аркадийной версии sqlalchemy диалект sqlite не поддерживает JSON
    def load_dialect_impl(self, dialect):
        if dialect.name == pg_dialect.name:
            return dialect.type_descriptor(JSONB())
        return dialect.type_descriptor(JSONEncodedDict())

    # https://docs.sqlalchemy.org/en/13/core/custom_types.html#overriding-type-compilation
    # Note that the behavior of coerce_compared_value is not inherited
    # by default from that of the base type.
    def coerce_compared_value(self, op, value):
        return self.impl.coerce_compared_value(op, value)
