# coding=utf-8
from __future__ import unicode_literals

from sqlalchemy import (
    Column, DateTime, ForeignKey, Index, Integer,
    String, UniqueConstraint, func, text
)
from sqlalchemy.ext.declarative import declarative_base

Base = declarative_base()


class DBInstanceMixin:
    id = Column('id', Integer, primary_key=True)
    created_at = Column('created_at', DateTime, server_default=func.now(), nullable=False)
    updated_at = Column('updated_at', DateTime, onupdate=func.now(), nullable=True)
    deleted_at = Column('deleted_at', DateTime, nullable=True)

    def __repr__(self):
        return '<{}: {}>'.format(
            self.__tablename__,
            repr(list({k: v for k, v in self.__dict__.items() if not k.startswith('_')}.items())),
        )


class UserAuthType(DBInstanceMixin, Base):
    __tablename__ = 'user_auth_type'
    name = Column('name', String, nullable=False, unique=True)


class UserAuth(DBInstanceMixin, Base):
    __tablename__ = 'user_auth'
    __table_args__ = (
        UniqueConstraint('user_auth_type_id', 'auth_value'),
    )
    user_auth_type_id = Column('user_auth_type_id', Integer, ForeignKey('user_auth_type.id'), nullable=False)
    auth_value = Column('auth_value', String, nullable=False)


class Email(DBInstanceMixin, Base):
    __tablename__ = 'email'
    __table_args__ = (
        Index('email_lower(email)_uindex', text('lower(email)'), unique=True),
        Index('emails_email_uindex', 'email', unique=True),
    )
    email = Column('email', String, unique=True)


class User(DBInstanceMixin, Base):
    __tablename__ = 'user'
    __table_args__ = (
        UniqueConstraint('user_auth_id', 'email_id'),
    )
    user_auth_id = Column('user_auth_id', Integer, ForeignKey('user_auth.id'), nullable=False)
    email_id = Column('email_id', Integer, ForeignKey('email.id'), nullable=False)
    name = Column('name', String, nullable=True)
    approved_at = Column('approved_at', DateTime, nullable=True)
    approving_token = Column('approving_token', String, nullable=True)
    user_timezone = Column('user_timezone', String, nullable=True)
