import logging.config
import os

import gunicorn.app.base
import sentry_sdk

from cloud_logger import default_config
from sentry_sdk.integrations.aiohttp import AioHttpIntegration
from sentry_sdk.integrations.logging import LoggingIntegration

import travel.avia.subscriptions.app.api.web as api_web
from travel.avia.subscriptions.app.settings.app import (
    AppConfig, Sentry, get_app_config
)

logger = logging.getLogger(__name__)


def setup_web():
    os.makedirs(os.environ['AVIA_LOG_PATH'], exist_ok=True)

    setup_logging()

    config: AppConfig = get_app_config()

    _configure_sentry(config.sentry, debug=config.debug)
    return api_web.WebApp(config=config)


def _ensure_key_exist(root: dict, key: str, factory: callable):
    value = root.get(key)
    if value is None:
        root[key] = factory()
        value = root[key]
    return value


def setup_logging():
    LOG_PATH = os.environ['AVIA_LOG_PATH']
    config = default_config(LOG_PATH)
    # Just in case someone modifies default_config
    formatters = _ensure_key_exist(config, 'formatters', dict)
    handlers = _ensure_key_exist(config, 'handlers', dict)
    loggers = _ensure_key_exist(config, 'loggers', dict)

    formatters['yt_simple'] = {
        'format': '%(message)s'
    }
    handlers['yt_simple'] = {
        'level': 'INFO',
        'class': 'cloud_logger.handlers.FileTreeHandler',
        'base_path': LOG_PATH,
        'formatter': 'yt_simple',
    }
    loggers['yt'] = {
        'handlers': ['yt_simple'],
        'level': 'INFO',
        'propagate': False,
    }
    logging.config.dictConfig(config)


def _configure_sentry(sentry: Sentry, debug: bool = False):
    if not sentry:
        return

    if sentry.FQDN and sentry.key and sentry.project:
        dsn = 'https://{}@{}/{}'.format(sentry.key, sentry.FQDN, sentry.project)
        logging.info('Setting up sentry %s', sentry)
        sentry_sdk.init(
            dsn=dsn,
            integrations=[AioHttpIntegration(), LoggingIntegration()],
            debug=debug,
            environment=os.environ.get('YANDEX_ENVIRONMENT_TYPE'),
            ca_certs=sentry.ca_certs,
        )


async def wsgi(*args, **kwargs):
    return setup_web().web_app


class GunicornAioApplication(gunicorn.app.base.Application):
    def init(self, parser, opts, args):
        pass

    def load_config(self):
        self.load_config_from_module_name_or_filename("python:travel.avia.subscriptions.app.gunicorn.conf")
        super().load_config()

    def load(self):
        return wsgi


def main():
    """
    Setup and parse arguments and run main
    """
    GunicornAioApplication().run()


if __name__ == '__main__':
    main()
