import os
import logging

from typing import Optional

import environ

from ticket_parser2.api.v1 import BlackboxClientId

from travel.avia.subscriptions.app.settings.config import for_environment, frontend_hosts_by_env
from travel.avia.subscriptions.app.settings.secrets import SecretStr, SettingsSecretResolver

logger = logging.getLogger(__name__)


def blackbox_client_converter(value):
    if isinstance(value, str):
        return BlackboxClientId(value)

    return value


def tvm_client_id_converter(value: Optional[str]):
    if value is not None and value.isnumeric():
        return int(value)

    return value


@environ.config
class DB:
    cluster = environ.var(converter=str)
    port = environ.var(converter=int)
    user = environ.var(converter=str)
    password = environ.var(converter=SecretStr)
    database = environ.var(converter=str)


@environ.config
class Sentry:
    key = environ.var(converter=SecretStr)
    FQDN = environ.var(default='', converter=str)
    project = environ.var(converter=str)
    ca_certs = environ.var(converter=str)


@environ.config
class Server:
    host = environ.var(default='0.0.0.0', converter=str)
    port = environ.var(default=8080, converter=int)


@environ.config
class Sender:
    auth_key = environ.var(converter=str)
    host = environ.var(converter=str)
    campaign_slug = environ.var(converter=str)
    account_slug = environ.var(converter=str)
    single_opt_in_campaign_slug = environ.var(converter=str)
    double_opt_in_campaign_slug = environ.var(converter=str)

    @property
    def auth(self):
        return self.auth_key, ''


@environ.config
class Business:
    force_user_confirm = environ.bool_var(default=True)  # RASPTICKETS-18487


@environ.config(prefix='')
class AppConfig:
    db = environ.group(DB)
    sentry = environ.group(Sentry)
    server = environ.group(Server)
    sender = environ.group(Sender)
    business = environ.group(Business)

    tvm_client_id = environ.var(default=None, converter=tvm_client_id_converter)
    blackbox_url = environ.var()
    blackbox_client = environ.var(converter=blackbox_client_converter)
    yav_token = environ.var(default=None, converter=SecretStr)
    sandbox_token = environ.var(default='', converter=SecretStr)
    mdb_token = environ.var(default=None, converter=SecretStr)
    debug = environ.var(default=False, converter=bool)


def get_app_config() -> AppConfig:
    config_dict = {
        **for_environment(os.environ.get('YANDEX_ENVIRONMENT_TYPE', 'dev')),
        **os.environ,
    }
    yav_token = config_dict.get('YAV_TOKEN')
    if yav_token:
        config_dict = SettingsSecretResolver(yav_token).resolve(config_dict)
    else:
        logger.warning('No yav token provided. Config secrets will not be resolved')

    app_config: AppConfig = environ.to_config(
        AppConfig,
        config_dict,
    )
    return app_config


def frontend_hosts():
    return frontend_hosts_by_env(os.environ.get('YANDEX_ENVIRONMENT_TYPE', 'dev'))
