import logging
from functools import lru_cache
from typing import Dict

from library.python.vault_client.instances import Production as VaultClient

log = logging.getLogger(__name__)


class SettingsSecretResolver:
    def __init__(self, yav_token: str, vault_client: type = VaultClient):
        self.yav_token: str = yav_token
        self.vault_client: type = vault_client

    def resolve(self, source_dict: dict) -> dict:
        if not source_dict:
            return source_dict
        output_dict = {}
        for k, v in source_dict.items():
            if isinstance(v, dict):
                output_dict[k] = self.resolve(v)
            elif isinstance(v, str) and v.startswith(('sec-', 'ver-')):
                value = self.secret_from_id(k, v)
                if value:
                    output_dict[k] = value
            else:
                output_dict[k] = v

        return output_dict

    def secret_from_id(self, owner: str, secret_id: str):
        if '.' in secret_id:
            sec_or_ver, exact_key = secret_id.split('.')
            return self.get_secret(self.yav_token, sec_or_ver, self.vault_client)[exact_key]

        secret = self.get_secret(self.yav_token, secret_id, self.vault_client)
        if len(secret) == 1:
            log.warning(
                'Exact secret key is not specified for secret `{}` of `{}`, '
                'but only one secret found: `{}`'.format(
                    secret_id, owner, list(secret.keys()),
                )
            )
            return next(iter(secret.values()))
        if len(secret) == 0:
            raise KeyError('Secret {} is empty.'.format(secret_id))

        raise KeyError(
            'Ambiguous secret {} definition ({} secrets found). '
            'Please specify exact secret key. Example: (sec-123.password)'.format(
                secret_id, len(secret),
            )
        )

    @staticmethod
    @lru_cache(maxsize=10)
    def get_secret(yandex_vault_token: str, secret_key: str, vault_client: type = VaultClient) -> Dict[str, str]:
        client = vault_client(
            authorization='OAuth {}'.format(yandex_vault_token),
            decode_files=True,
        )
        return client.get_version(secret_key)['value']


class SecretStr(str):
    """
    String that censors its __repr__ if called from an attrs repr.
    """

    def __repr__(self):
        return "<SECRET>"
