package main

import (
	"bytes"
	"context"
	"fmt"
	"text/template"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/avia/suggests/api/logging"
)

type LanguagePair struct {
	from string
	to   string
}

type (
	Config struct {
		Env string `config:"YANDEX_ENVIRONMENT_TYPE,backend=env,required"`

		Engine   Engine
		Features Features
		Listen   Listen
		Logging  logging.Logging
		Mysql    Mysql
		Services Services
		Probe    *probes.HTTPConfig
	}

	Engine struct {
		DefaultSuggests int   `config:"default_number_of_suggests,backend=yaml" yaml:"default_number_of_suggests"`
		MaxSuggest      int   `config:"max_suggest,backend=yaml" yaml:"max_suggest"`
		CacheSize       int64 `config:"cache_size,backend=yaml" yaml:"cache_size"`
		CGSize          int   `config:"gc_size,backend=yaml" yaml:"gc_size"`
		MaxTypoDistance int   `config:"max_typo_distance,backend=yaml" yaml:"max_typo_distance"`
		MinTypoDistance int   `config:"min_typo_distance,backend=yaml" yaml:"min_typo_distance"`
		MinLenForTypo   int   `config:"min_len_for_typo,backend=yaml" yaml:"min_len_for_typo"`

		// По языку список языков, на которых искать саджесты
		SuggestLanguages map[string][]string `config:"languages_main,backend=yaml" yaml:"languages_main"`

		// По языку список языков, на которых искать саджесты с опечатками
		SuggestLanguagesTypos map[string][]string `config:"languages_typos,backend=yaml" yaml:"languages_typos"`

		// Как переключать раскладку
		PuntoLanguagesRaw map[string][]string `config:"languages_punto,backend=yaml" yaml:"languages_punto"`
		PuntoLanguages    map[string][]LanguagePair
	}

	Features struct {
		UseGeobase         bool `config:"use_geobase,backend=yaml" yaml:"use_geobase"`
		ShowHiddenAirports bool `config:"show_hidden_airports,backend=yaml" yaml:"show_hidden_airports"`
	}

	Services struct {
		PriceIndexURL string `config:"AVIA_PRICE_INDEX_URL,backend=env,required"`
	}

	Listen struct {
		Host string `config:"host,backend=yaml"`
		Port int    `config:"port,backend=yaml"`
	}

	Mysql struct {
		Host      string `config:"AVIA_MYSQL_HOST,backend=env"`
		ClusterID string `config:"AVIA_MYSQL_CLUSTER_ID,backend=env"`
		DBName    string `config:"AVIA_MYSQL_DB_NAME,backend=env" yaml:"dbname" valid:"required"`
		Password  string `config:"AVIA_MYSQL_PASSWORD,backend=env,required"`
		Port      int    `config:"AVIA_MYSQL_PORT,backend=env" yaml:"port" valid:"required"`
		Protocol  string `config:"AVIA_MYSQL_PROTOCOL,backend=env" yaml:"protocol" valid:"required"`
		UserName  string `config:"AVIA_MYSQL_USER,backend=env" yaml:"user" valid:"required"`
	}
)

func ReadConfig(path string) (*Config, error) {
	var config Config
	config.Probe = &probes.DefaultHTTPConfig

	var backends []backend.Backend

	backends = append(backends, file.NewBackend(path))
	backends = append(backends, env.NewBackend())

	err := confita.NewLoader(backends...).Load(context.Background(), &config)
	if err != nil {
		return nil, fmt.Errorf("fatal error config file (%s): %v", path, err)
	}

	if config.Mysql.Host == "" {
		config.Mysql.Host = fmt.Sprintf("c-%s.ro.db.yandex.net", config.Mysql.ClusterID)
	}

	config.Engine.PuntoLanguages = make(map[string][]LanguagePair)
	for l, puntoList := range config.Engine.PuntoLanguagesRaw {
		config.Engine.PuntoLanguages[l] = make([]LanguagePair, 0)
		for _, otherLang := range puntoList {
			config.Engine.PuntoLanguages[l] = append(
				config.Engine.PuntoLanguages[l],
				LanguagePair{l, otherLang},
				LanguagePair{otherLang, l},
			)
		}
	}

	return &config, nil
}

func (d *Mysql) ConnectionString() (string, error) {
	var connectionString bytes.Buffer

	DsnTemplate := "{{.UserName}}:{{.Password}}@{{.Protocol}}({{.Host}}:{{.Port}})/{{.DBName}}?charset=utf8&parseTime=true&loc=Local"

	t := template.New("DSN template")
	t, err := t.Parse(DsnTemplate)
	if err != nil {
		return "", fmt.Errorf("can not parse db dsn string: %v", err)
	}

	err = t.Execute(&connectionString, d)
	if err != nil {
		return "", fmt.Errorf("can not fill template: %w", err)
	}

	return connectionString.String(), nil
}
