package main

import "sort"

func max(nums ...int) int {
	ans := 0
	for ind, number := range nums {
		if ind == 0 || ans < number {
			ans = number
		}
	}
	return ans
}

func min(nums ...int) int {
	ans := 0
	for ind, number := range nums {
		if ind == 0 || ans > number {
			ans = number
		}
	}
	return ans
}

func abs(a int) int {
	if a < 0 {
		return -a
	}
	return a
}

func makeZeroMatrix(height int, width int) [][]int {
	var matrix = make([][]int, height)
	for ind := range matrix {
		matrix[ind] = make([]int, width)
	}
	return matrix
}

// Damerau–Levenshtein distance
// See https://en.wikipedia.org/wiki/Damerau–Levenshtein_distance
func DamLevDistance(first, second []rune) int {
	firstLen := len(first)
	secondLen := len(second)

	if firstLen < secondLen {
		first, second = second, first
		firstLen, secondLen = secondLen, firstLen
	}

	if (firstLen == 0) || (secondLen == 0) {
		return max(firstLen, secondLen)
	}

	distMatrix := makeZeroMatrix(3, secondLen+1)
	rowTwoBehind := 0
	rowOneBehind := 1
	row := 2

	// Filling zero row
	for j := 1; j <= secondLen; j++ {
		distMatrix[rowOneBehind][j] = j
	}

	for i := 1; i <= firstLen; i++ {
		firstStringInd := i - 1
		distMatrix[row][0] = i
		for j := 1; j <= secondLen; j++ {
			secondStringInd := j - 1

			additional := 0
			if first[firstStringInd] != second[secondStringInd] {
				additional = 1
			}

			dist := min(distMatrix[rowOneBehind][j]+1, distMatrix[row][j-1]+1, distMatrix[rowOneBehind][j-1]+additional)
			if (i > 1) &&
				(j > 1) &&
				(first[firstStringInd] == second[secondStringInd-1]) &&
				(first[firstStringInd-1] == second[secondStringInd]) {
				dist = min(dist, distMatrix[rowTwoBehind][j-2]+1)
			}
			distMatrix[row][j] = dist
		}

		rowTwoBehind, rowOneBehind, row = rowOneBehind, row, rowTwoBehind
	}

	return distMatrix[rowOneBehind][secondLen]
}

func CheckDamLevDistance(first, second []rune, maxDistance int, distMatrix [][]int) int {
	firstLen := len(first)
	secondLen := len(second)

	if firstLen < secondLen {
		first, second = second, first
		firstLen, secondLen = secondLen, firstLen
	}

	if (firstLen == 0) || (secondLen == 0) {
		ans := max(firstLen, secondLen)
		if maxDistance >= 0 && ans > maxDistance {
			return -1
		}

		return ans

	}

	rowTwoBehind := 0
	rowOneBehind := 1
	row := 2

	minRowOneBehind := 0
	minRow := 0

	// Filling zero row
	for j := 0; j <= secondLen; j++ {
		distMatrix[rowOneBehind][j] = j
	}

	for i := 1; i <= firstLen; i++ {
		firstStringInd := i - 1
		distMatrix[row][0] = i
		minRow = i
		for j := 1; j <= secondLen; j++ {
			secondStringInd := j - 1

			additional := 0
			if first[firstStringInd] != second[secondStringInd] {
				additional = 1
			}

			dist := min(distMatrix[rowOneBehind][j]+1, distMatrix[row][j-1]+1, distMatrix[rowOneBehind][j-1]+additional)
			if (i > 1) &&
				(j > 1) &&
				(first[firstStringInd] == second[secondStringInd-1]) &&
				(first[firstStringInd-1] == second[secondStringInd]) {
				dist = min(dist, distMatrix[rowTwoBehind][j-2]+1)
			}
			distMatrix[row][j] = dist
			minRow = min(minRow, dist)
		}

		if maxDistance > 0 && min(minRow, minRowOneBehind) > maxDistance {
			return -1
		}

		rowTwoBehind, rowOneBehind, row = rowOneBehind, row, rowTwoBehind
		minRowOneBehind = minRow
	}

	answer := distMatrix[rowOneBehind][secondLen]
	if answer <= maxDistance {
		return answer
	}
	return -1
}

type WordWithDistance struct {
	Word     []rune
	Distance int
}

type ByDistance []WordWithDistance

func (tokens ByDistance) Len() int           { return len(tokens) }
func (tokens ByDistance) Swap(i, j int)      { tokens[i], tokens[j] = tokens[j], tokens[i] }
func (tokens ByDistance) Less(i, j int) bool { return tokens[i].Distance < tokens[j].Distance }

func FindNearest(word []rune, dictionary [][]rune, maxDistance int) []WordWithDistance {
	if len(dictionary) == 0 {
		return []WordWithDistance{}
	}

	distances := make([]WordWithDistance, len(dictionary))
	distMatrix := makeZeroMatrix(3, len(dictionary[len(dictionary)-1])+len(word)+1)
	foundWords := 0
	for _, targetWord := range dictionary {
		// Optimization
		if (maxDistance > 0) && (abs(len(word)-len(targetWord)) > maxDistance) {
			continue
		}
		distance := CheckDamLevDistance(word, targetWord, maxDistance, distMatrix)
		if maxDistance < 0 || distance > 0 {
			distances[foundWords] = WordWithDistance{
				Word:     targetWord,
				Distance: distance,
			}
			foundWords++
		}
	}

	distancesShrink := distances[:foundWords]
	sort.Sort(ByDistance(distancesShrink))

	return distancesShrink
}
