package main

import (
	"fmt"
	"net/http"
	"net/url"
	"sync/atomic"
	"time"

	"github.com/labstack/echo/v4"
)

func IndexV1(c echo.Context) error {
	return Index(c, false)
}

func IndexV2(c echo.Context) error {
	return Index(c, true)
}

func Index(ctx echo.Context, blended bool) error {
	defer func() {
		if r := recover(); r != nil {
			uri := ctx.QueryString()
			message := fmt.Sprintf("Recovered Index: %v, URI: %s", r, uri)
			appLogger.Error(message)
		}
	}()

	startTime := time.Now()
	var response string
	var cacheHitMiss string
	var cs *CachedSuggests
	var count int

	q := Query{}
	q.FillFromRequest(ctx, blended)

	if q.geobaseID != 0 {
		// There is no reason to cache such queries
		suggests := sf.FindSuggestsByGeobaseID(q)
		response = sf.MakeResponse(suggests, q)
		count = len(suggests)
	} else {
		lruCacheKey := q.Key()
		cachedItem := SuggestLRUCache.Get(lruCacheKey)

		if cachedItem == nil {
			cacheHitMiss = "MIS"
			suggests := sf.FindSuggests(q.rawQuery, q)

			needCountry := q.needCountry && (q.field == "to")
			if blended {
				suggests = sf.FilterByPointType(suggests, needCountry, q.needStation)
			} else {
				suggests = sf.FilterAndSortByPointType(suggests, needCountry, q.needStation)
			}

			if q.flattenSuggests {
				for i := range suggests {
					suggests[i].level = 0
				}
			}

			response = sf.MakeResponse(suggests, q)

			cs = &CachedSuggests{
				PlainText: response,
				Count:     len(suggests),
			}

			// Не кешируем вариант до момента загрузки данных
			if sf.SuccessLoaded {
				SuggestLRUCache.Set(lruCacheKey, cs, time.Minute*60*24)
				atomic.AddInt64(&SuggestStat.CacheMiss, 1)
			}

			count = len(suggests)

		} else {
			cacheHitMiss = "HIT"
			cs = cachedItem.Value().(*CachedSuggests)
			response = cs.PlainText
			atomic.AddInt64(&SuggestStat.CacheHit, 1)
			count = cs.Count
		}
	}

	if q.callbackName != "" {
		// JSONP
		ctx.Response().Header().Set(echo.HeaderContentType, echo.MIMEApplicationJavaScriptCharsetUTF8)
		ctx.Response().Header().Set(echo.HeaderXContentTypeOptions, "nosniff")
		ctx.Response().Header().Set(echo.HeaderContentDisposition, "attachment; filename=json.txt")

		response = "/**/" + q.callbackName + "(" + response + ");"
	} else {
		// JSON
		origin := ctx.Request().Header.Get("Origin")
		originURL, err := url.Parse(origin)
		// чтобы не яндексовые сайты не пользовались нашими саггестами
		if err == nil && len(origin) != 0 {
			var allowHost string
			if checkOrigin(originURL.Host) {
				allowHost = originURL.Scheme + "://" + originURL.Host
				ctx.Response().Header().Set(echo.HeaderAccessControlAllowOrigin, allowHost)
			} else {
				response = ""
			}
		}
		ctx.Response().Header().Set(echo.HeaderContentType, echo.MIMEApplicationJSONCharsetUTF8)
	}

	q.Log(cacheHitMiss, time.Since(startTime), count)

	return ctx.String(http.StatusOK, response)
}

func checkOrigin(host string) bool {
	return originRegexp.Match([]byte(host))
}

func CacheStat(ctx echo.Context) error {
	var cacheHitRatio float64

	cacheHit := float64(SuggestStat.CacheHit)
	cacheMiss := float64(SuggestStat.CacheMiss)
	cacheTotal := cacheHit + cacheMiss

	if cacheTotal > 0 {
		cacheHitRatio = cacheHit / cacheTotal * 100
	} else {
		cacheHitRatio = 0
	}

	response := fmt.Sprintf("cache_hit: %f\n", cacheHit)
	response = response + fmt.Sprintf("cache_miss: %f\n", cacheMiss)
	response = response + fmt.Sprintf("cache_hit_ratio: %6.2f%%", cacheHitRatio)

	return ctx.String(http.StatusOK, response)
}
