package main

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strconv"
	"strings"
	"time"

	_ "github.com/go-sql-driver/mysql"
)

func NewConnection() (*sql.DB, error) {
	// Идем на случайный слейв
	DsnString, err := config.Mysql.ConnectionString()
	if err != nil {
		return nil, fmt.Errorf("can not create connection: %w", err)
	}
	return sql.Open("mysql", DsnString)
}

// IATA
func LoadIATA(con *sql.DB) (map[string]string, error) {
	m := make(map[string]string)
	appLogger.Info("Load IATA codes")

	rows, err := con.Query("SELECT s.id, COALESCE(s.settlement_id, 0), sc.code FROM www_station s JOIN www_stationcode sc ON s.id = sc.station_id WHERE sc.system_id=? and s.majority_id <= ?", 4, MaxStationMajority)
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var pointID int
		var settlementID int
		var code string

		err = rows.Scan(&pointID, &settlementID, &code)
		if err != nil {
			return nil, err
		}

		key := "s" + strconv.Itoa(pointID)
		m[key] = code

		if settlementID > 0 {
			key := "c" + strconv.Itoa(settlementID)
			m[key] = code
		}
	}

	rows, err = con.Query("SELECT id, COALESCE(iata, '') FROM www_settlement")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var pointID int
		var code string

		err = rows.Scan(&pointID, &code)
		if err != nil {
			return nil, err
		}

		if code != "" {
			settlementKey := "c" + strconv.Itoa(pointID)
			m[settlementKey] = code
		}

	}

	return m, nil
}

// IATA
func LoadSirena(con *sql.DB) (map[string]string, error) {
	m := make(map[string]string)

	appLogger.Info("Load Sirena codes")

	rows, err := con.Query("SELECT s.id, COALESCE(s.settlement_id, 0), sc.code FROM www_station s JOIN www_stationcode sc ON s.id = sc.station_id WHERE sc.system_id=? and s.majority_id <= ?", 1, MaxStationMajority)
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var pointID int
		var settlementID int
		var code string

		err = rows.Scan(&pointID, &settlementID, &code)
		if err != nil {
			return nil, err
		}

		key := "s" + strconv.Itoa(pointID)
		m[key] = code

		if settlementID > 0 {
			key := "c" + strconv.Itoa(settlementID)
			m[key] = code
		}
	}

	return m, nil
}

// Points
func LoadPoints(con *sql.DB) (map[string]Point, error) {
	m := make(map[string]Point)
	var settlementIDs []string

	isSettlementHidden := make(map[string]bool)

	appLogger.Info("Load Points")

	rows, err := con.Query("SELECT s.id, s.settlement_id, s.title, t.ru_nominative, t.tr_nominative, t.uk_nominative, t.en_nominative, t.de_nominative, s.country_id, s.region_id, s.hidden FROM www_station as s JOIN www_translated_title AS t ON new_L_title_id = t.id WHERE t_type_id=2 and s.majority_id <= ?", MaxStationMajority)
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var (
			pointID      sql.NullInt64
			settlementID sql.NullInt64
			title        sql.NullString
			titleRu      sql.NullString
			titleTr      sql.NullString
			titleUk      sql.NullString
			titleEn      sql.NullString
			titleDe      sql.NullString
			countryID    sql.NullInt64
			regionID     sql.NullInt64
			hidden       sql.NullBool
		)

		err = rows.Scan(&pointID, &settlementID, &title, &titleRu, &titleTr, &titleUk, &titleEn, &titleDe, &countryID, &regionID, &hidden)
		if err != nil {
			return nil, err
		}

		p := Point{
			title:     title.String,
			pointKey:  "s" + strconv.Itoa(int(pointID.Int64)),
			regionID:  int(regionID.Int64),
			countryID: int(countryID.Int64),
			titleByLang: map[string]string{
				"ru": titleRu.String,
				"uk": titleUk.String,
				"tr": titleTr.String,
				"en": titleEn.String,
				"de": titleDe.String,
			},
			hidden:               hidden.Bool,
			haveAirport:          true,
			haveNotHiddenAirport: !hidden.Bool,
		}

		m[p.pointKey] = p

		settlementIDStr := "c" + strconv.Itoa(int(settlementID.Int64))
		settlementIDs = append(settlementIDs, strconv.Itoa(int(settlementID.Int64)))

		status, ok := isSettlementHidden[settlementIDStr]
		if ok {
			if status { // У settlement не было ни одной открытой станции
				isSettlementHidden[settlementIDStr] = hidden.Bool
			} // Иначе ничего не делаем, потому что уже нашли открытую станцию
		} else { // Первая станция для settlement
			isSettlementHidden[settlementIDStr] = hidden.Bool
		}
	}

	rows, err = con.Query("SELECT station_id, settlement_id FROM www_station2settlement")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var stationID sql.NullInt64
		var settlementID sql.NullInt64

		err = rows.Scan(&stationID, &settlementID)
		if err != nil {
			return nil, err
		}

		stationKey := "s" + strconv.Itoa(int(stationID.Int64))
		if _, ok := m[stationKey]; ok {
			settlementIDStr := "c" + strconv.Itoa(int(settlementID.Int64))
			settlementIDs = append(settlementIDs, strconv.Itoa(int(settlementID.Int64)))
			status, ok := isSettlementHidden[settlementIDStr]
			if ok {
				if status { // У settlement не было ни одной открытой станции
					isSettlementHidden[settlementIDStr] = m[stationKey].hidden
				} // Иначе ничего не делаем, потому что уже нашли открытую станцию
			} else { // Первая станция для settlement
				isSettlementHidden[settlementIDStr] = m[stationKey].hidden
			}
		}
	}

	settlementIdsStr := strings.Join(settlementIDs, ",")
	query := "SELECT s.id, s.title, t.ru_nominative, t.tr_nominative, t.uk_nominative, t.en_nominative, t.de_nominative, s.country_id, s.region_id, s._geo_id, s.hidden FROM www_settlement as s JOIN www_translated_title AS t ON new_L_title_id = t.id WHERE s.majority_id < 5 OR s.majority_id = 99"
	if len(settlementIdsStr) > 0 {
		query += " OR s.id IN (" + settlementIdsStr + ")"
	}
	rows, err = con.Query(query)
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var (
			pointID     sql.NullInt64
			title       sql.NullString
			titleRu     sql.NullString
			titleTr     sql.NullString
			titleUk     sql.NullString
			titleEn     sql.NullString
			titleDe     sql.NullString
			countryID   sql.NullInt64
			regionID    sql.NullInt64
			geoID       sql.NullInt64
			hidden      sql.NullBool
			titleUkStr  string
			haveAirport bool
		)

		err = rows.Scan(&pointID, &title, &titleRu, &titleTr, &titleUk, &titleEn, &titleDe, &countryID, &regionID, &geoID, &hidden)
		if err != nil {
			return nil, err
		}

		if titleUk.String == "" {
			titleUkStr, err = GeobaseTitle(titleUk.String, int(geoID.Int64), "uk")
			if err != nil {
				return nil, err
			}
		} else {
			titleUkStr = titleUk.String
		}

		settlementIDStr := "c" + strconv.Itoa(int(pointID.Int64))
		allAirportsAreHidden, ok := isSettlementHidden[settlementIDStr]
		if ok { // У города есть аэропорт, скрытый или нет
			haveAirport = true
		} else {
			haveAirport = false
		}

		p := Point{
			title:     title.String,
			pointKey:  settlementIDStr,
			regionID:  int(regionID.Int64),
			countryID: int(countryID.Int64),
			titleByLang: map[string]string{
				"ru": titleRu.String,
				"uk": titleUkStr,
				"tr": titleTr.String,
				"en": titleEn.String,
				"de": titleDe.String,
			},
			hidden:               hidden.Bool || isSettlementHidden[settlementIDStr],
			haveAirport:          haveAirport,
			haveNotHiddenAirport: !allAirportsAreHidden && ok,
		}

		m[p.pointKey] = p
	}

	rows, err = con.Query("SELECT c.id, c.title, t.ru_nominative, t.tr_nominative, t.uk_nominative, t.en_nominative, t.de_nominative FROM www_country as c JOIN www_translated_title AS t ON new_L_title_id = t.id")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var (
			pointID    sql.NullInt64
			title      sql.NullString
			titleRu    sql.NullString
			titleTr    sql.NullString
			titleUk    sql.NullString
			titleUkStr string
			titleEn    sql.NullString
			titleDe    sql.NullString
		)

		err = rows.Scan(&pointID, &title, &titleRu, &titleTr, &titleUk, &titleEn, &titleDe)
		if err != nil {
			return nil, err
		}

		if titleUk.String == "" {
			titleUkStr, err = GeobaseTitle(titleUk.String, int(pointID.Int64), "uk")
			if err != nil {
				return nil, err
			}
		} else {
			titleUkStr = titleUk.String
		}

		p := Point{
			title:     title.String,
			countryID: int(pointID.Int64),
			pointKey:  "l" + strconv.Itoa(int(pointID.Int64)),
			titleByLang: map[string]string{
				"ru": titleRu.String,
				"uk": titleUkStr,
				"tr": titleTr.String,
				"en": titleEn.String,
				"de": titleDe.String,
			},
			hidden:               false, // Страны не бывают скрытыми
			haveAirport:          true,
			haveNotHiddenAirport: true,
		}

		m[p.pointKey] = p
	}

	return m, nil

}

// Regions
func LoadRegions(con *sql.DB) (map[int]Region, error) {
	m := make(map[int]Region)

	appLogger.Info("Load regions")
	rows, err := con.Query("SELECT r.id, r.title, t.ru_nominative, t.tr_nominative, t.uk_nominative, t.en_nominative, t.de_nominative FROM www_region AS r JOIN www_translated_title AS t ON new_L_title_id = t.id")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var (
			id      sql.NullInt64
			title   sql.NullString
			titleRu sql.NullString
			titleTr sql.NullString
			titleUk sql.NullString
			titleEn sql.NullString
			titleDe sql.NullString
		)

		err = rows.Scan(&id, &title, &titleRu, &titleTr, &titleUk, &titleEn, &titleDe)
		if err != nil {
			return nil, err
		}

		r := Region{
			id:    int(id.Int64),
			title: title.String,
			titleByLang: map[string]string{
				"ru": titleRu.String,
				"uk": titleUk.String,
				"tr": titleTr.String,
				"en": titleEn.String,
				"de": titleEn.String,
			},
		}

		m[r.id] = r
	}

	return m, nil

}

// Countries
func LoadCountries(con *sql.DB) (map[int]Country, error) {
	m := make(map[int]Country)

	appLogger.Info("Load countries")
	rows, err := con.Query("SELECT c.id, c.code, c.title, t.ru_nominative, t.tr_nominative, t.uk_nominative, t.en_nominative, t.de_nominative FROM www_country as c JOIN www_translated_title AS t ON new_L_title_id = t.id")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var (
			id         sql.NullInt64
			code       sql.NullString
			title      sql.NullString
			titleRu    sql.NullString
			titleTr    sql.NullString
			titleUk    sql.NullString
			titleUkStr string
			titleEn    sql.NullString
			titleDe    sql.NullString
		)

		err = rows.Scan(&id, &code, &title, &titleRu, &titleTr, &titleUk, &titleEn, &titleDe)
		if err != nil {
			return nil, err
		}

		if titleUk.String == "" {
			titleUkStr, err = GeobaseTitle(titleUk.String, int(id.Int64), "uk")
			if err != nil {
				return nil, err
			}
		} else {
			titleUkStr = titleUk.String
		}

		c := Country{
			id:    int(id.Int64),
			title: title.String,
			code:  strings.ToLower(code.String),
			titleByLang: map[string]string{
				"ru": titleRu.String,
				"uk": titleUkStr,
				"tr": titleTr.String,
				"en": titleEn.String,
				"de": titleDe.String,
			},
		}

		m[c.id] = c
	}

	return m, nil
}

// Популярность направлений для приоритезации (логи + расписание рейсов)
func LoadPopularDirections(con *sql.DB) (map[string]int, error) {
	m := make(map[string]int)

	// Regions
	appLogger.Info("Load popular directions")
	rows, err := con.Query("SELECT departure_settlement_id, arrival_settlement_id, popularity, national_version FROM avia_aviadirectionnational")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var departureSettlementID int
		var arrivalSettlementID int
		var popularity int
		var nationalVersion string

		err = rows.Scan(&departureSettlementID, &arrivalSettlementID, &popularity, &nationalVersion)
		if err != nil {
			return nil, err
		}

		key := fmt.Sprintf("%v_c%v_c%v", nationalVersion, departureSettlementID, arrivalSettlementID)
		m[key] = popularity
	}

	// Строим популярность Город -> Страна
	// Для каждого города вылета находим самое популярное направление в страну прилета и берем его популярность
	rows, err = con.Query(
		`
		SELECT
			departure_settlement_id, s.country_id, MAX(popularity), national_version
		FROM
			avia_aviadirectionnational
		JOIN www_settlement AS s ON s.id = arrival_settlement_id

		GROUP BY
			departure_settlement_id, s.country_id, national_version
		`,
	)
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var departureSettlementID int
		var countryID int
		var popularity int
		var nationalVersion string

		err = rows.Scan(&departureSettlementID, &countryID, &popularity, &nationalVersion)
		if err != nil {
			return nil, err
		}

		countryKey := fmt.Sprintf("%v_c%v_l%v", nationalVersion, departureSettlementID, countryID)
		m[countryKey] = popularity
	}

	return m, nil

}

func loadPopularSettlementsByCountries(con *sql.DB) (map[string][]SuggestMetaInfo, error) {
	m := make(map[string][]SuggestMetaInfo)

	appLogger.Info("Load popular settlements")
	rows, err := con.Query(`
		SELECT
			settlement_id, t.country_id AS country_id, popularity
		FROM avia_aviasettlement

		INNER JOIN www_settlement AS t ON settlement_id = t.id

		WHERE
			arrival = 0
		GROUP BY settlement_id, country_id
		ORDER BY country_id, popularity DESC
	`)
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var settlementID sql.NullInt64
		var countryID sql.NullInt64
		var popularity sql.NullInt64

		err = rows.Scan(&settlementID, &countryID, &popularity)
		if err != nil {
			return nil, err
		}

		if !settlementID.Valid || !countryID.Valid || !popularity.Valid {
			continue
		}

		pointKey := fmt.Sprintf("c%v", settlementID.Int64)
		countryKey := fmt.Sprintf("l%v", countryID.Int64)

		sm := SuggestMetaInfo{
			pointKey:       pointKey,
			pointTypeOrder: SettlementType,
			popularity:     int(popularity.Int64),
		}

		settlementIDs, ok := m[countryKey]

		if ok {
			if len(settlementIDs) > config.Engine.MaxSuggest+5 {
				continue
			}
		}

		m[countryKey] = append(m[countryKey], sm)
	}

	return m, nil
}

func loadTop15Settlements(con *sql.DB) (map[string][]SuggestMetaInfo, error) {
	m := make(map[string][]SuggestMetaInfo)

	// Regions
	appLogger.Info("Load TOP-15 settlements")
	rows, err := con.Query("SELECT settlement_id, national_version FROM avia_aviasettlementnational GROUP BY settlement_id, national_version ORDER by national_version, popularity DESC")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var settlementID sql.NullInt64
		var nationalVersion sql.NullString

		err = rows.Scan(&settlementID, &nationalVersion)
		if err != nil {
			return nil, err
		}

		if !settlementID.Valid || !nationalVersion.Valid {
			continue
		}

		pointKey := fmt.Sprintf("c%v", settlementID.Int64)

		sm := SuggestMetaInfo{
			pointKey: pointKey,
		}

		settlementIDs, ok := m[nationalVersion.String]

		if ok {
			if len(settlementIDs) > config.Engine.MaxSuggest+5 {
				continue
			}
		}

		m[nationalVersion.String] = append(m[nationalVersion.String], sm)
	}

	return m, nil
}

func LoadTop15(con *sql.DB, pointMap map[string]Point, countryMap map[int]Country) (map[string][]SuggestMetaInfo, error) {
	m := make(map[string][]SuggestMetaInfo)

	// Regions
	appLogger.Info("Load TOP-15 directions")
	rows, err := con.Query("SELECT departure_settlement_id, arrival_settlement_id, popularity, national_version FROM avia_aviadirectionnational ORDER BY national_version, departure_settlement_id, popularity DESC")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var departureSettlementID int
		var arrivalSettlementID int
		var popularity int
		var nationalVersion string

		err = rows.Scan(&departureSettlementID, &arrivalSettlementID, &popularity, &nationalVersion)
		if err != nil {
			return nil, err
		}

		// В украинской версии не должно быть российских городов в деф. саджесте
		arrivalSettlementPointKey := "c" + strconv.Itoa(arrivalSettlementID)
		arrivalPoint := pointMap[arrivalSettlementPointKey]
		arrivalCountry := countryMap[arrivalPoint.countryID]

		if nationalVersion == "ua" && arrivalCountry.code == "RU" {
			continue
		}

		key := fmt.Sprintf("%v_c%v", nationalVersion, departureSettlementID)

		sm := SuggestMetaInfo{
			pointKey: arrivalSettlementPointKey,
		}

		settlementIDs, ok := m[key]

		if ok {
			if len(settlementIDs) > config.Engine.MaxSuggest+5 {
				continue
			}
		}

		m[key] = append(m[key], sm)
	}

	return m, nil
}

// Связь станций и городов
func LoadStations(con *sql.DB, takeHiddenAirports bool) (map[string]string, error) {
	m := make(map[string]string)
	var query string
	if takeHiddenAirports {
		query = "SELECT id, COALESCE(settlement_id, 0) FROM www_station WHERE t_type_id=2 and majority_id <= ?"
	} else {
		query = "SELECT id, COALESCE(settlement_id, 0) FROM www_station WHERE t_type_id=2 AND hidden = 0 and majority_id <= ?"
	}
	rows, err := con.Query(query, MaxStationMajority)
	if err != nil {
		return nil, err
	}
	for rows.Next() {
		var stationID int
		var settlementID int

		err = rows.Scan(&stationID, &settlementID)
		if err != nil {
			return nil, err
		}

		stationKey := "s" + strconv.Itoa(stationID)
		settlementKey := "c" + strconv.Itoa(settlementID)

		m[stationKey] = settlementKey
	}

	return m, nil
}

// Популярность города/аэропорта
func LoadPopularity(con *sql.DB) (map[string]int, error) {
	m := make(map[string]int)

	appLogger.Info("Load popularity for cities")
	rows, err := con.Query("select settlement_id, popularity, national_version from avia_aviasettlementnational where arrival=0")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var settlementID int
		var popularity int
		var nationalVersion string

		err = rows.Scan(&settlementID, &popularity, &nationalVersion)
		if err != nil {
			return nil, err
		}

		key := fmt.Sprintf("%v_c%v", nationalVersion, settlementID)
		m[key] = popularity
	}

	appLogger.Info("Load popularity for stations")
	rows, err = con.Query(
		`
			SELECT
				station.id, popularity_table.popularity, popularity_table.national_version
			FROM
				www_station as station
			INNER JOIN
				avia_aviasettlementnational AS popularity_table
			ON popularity_table.settlement_id = station.settlement_id
			WHERE popularity_table.arrival = 0 and station.majority_id <= ?
		`, MaxStationMajority)

	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var stationID int
		var popularity int
		var nationalVersion string

		err = rows.Scan(&stationID, &popularity, &nationalVersion)
		if err != nil {
			return nil, err
		}

		key := fmt.Sprintf("%v_s%v", nationalVersion, stationID)
		m[key] = popularity
	}

	rows, err = con.Query(
		`
			SELECT
				station.id, popularity_table.popularity, popularity_table.national_version
			FROM
				www_station as station
			INNER JOIN
				www_station2settlement AS s2s
			ON s2s.station_id = station.id
			INNER JOIN
				avia_aviasettlementnational AS popularity_table
			ON popularity_table.settlement_id = s2s.settlement_id
			WHERE popularity_table.arrival = 0 and station.majority_id <= ?
		`, MaxStationMajority)

	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var stationID int
		var popularity int
		var nationalVersion string

		err = rows.Scan(&stationID, &popularity, &nationalVersion)
		if err != nil {
			return nil, err
		}

		key := fmt.Sprintf("%v_s%v", nationalVersion, stationID)
		m[key] = max(popularity, m[key])
	}

	return m, nil
}

// Размер города/аэропорта
func LoadMajority(con *sql.DB) (map[string]int, error) {
	m := make(map[string]int)

	appLogger.Info("Load majority")
	rows, err := con.Query("select id, majority_id FROM www_settlement")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var settlementID int
		var popularity *int

		err = rows.Scan(&settlementID, &popularity)
		if err != nil {
			return nil, err
		}

		pointKey := "c" + strconv.Itoa(settlementID)
		m[pointKey] = UnknownSettlementMajority
		if popularity != nil {
			m[pointKey] = *popularity
		}
	}

	// Сложим сразу и для станций, что бы потом лишний раз не преобразовывать станцию в город
	rows, _ = con.Query("select id, settlement_id FROM www_station WHERE t_type_id=2 AND settlement_id > 0 AND majority_id <= ?", MaxStationMajority)
	for rows.Next() {
		var stationID int
		var settlementID int

		err = rows.Scan(&stationID, &settlementID)
		if err != nil {
			return nil, err
		}

		stationKey := "s" + strconv.Itoa(stationID)
		settlementKey := "c" + strconv.Itoa(settlementID)

		majority, ok := m[settlementKey]

		if ok {
			m[stationKey] = majority
		}
	}

	return m, nil
}

func LoadSuggests(sf *SuggestFabric, con *sql.DB, takeHiddenAirports bool) error {
	// Load stations
	appLogger.Info("Load and index stations")
	// FIXME: t_type_id hardcoded
	var query string
	if takeHiddenAirports {
		query = "SELECT s.id, s.settlement_id, t.ru_nominative, t.tr_nominative, t.uk_nominative, t.en_nominative, t.de_nominative FROM www_station as s JOIN www_translated_title AS t ON new_L_title_id = t.id WHERE t_type_id=2 and s.majority_id <= ?"
	} else {
		query = `
			SELECT
				s.id, s.settlement_id, t.ru_nominative, t.tr_nominative, t.uk_nominative, t.en_nominative, t.de_nominative
			FROM www_station as s
			JOIN www_translated_title AS t ON new_L_title_id = t.id
			WHERE t_type_id = 2 AND hidden = 0 and s.majority_id <= ?
		`
	}
	rows, err := con.Query(query, MaxStationMajority)
	if err != nil {
		return err
	}

	for rows.Next() {
		var (
			pointID      sql.NullInt64
			settlementID sql.NullInt64
			titleRu      sql.NullString
			titleUa      sql.NullString
			titleTr      sql.NullString
			titleEn      sql.NullString
			titleDe      sql.NullString
		)

		err = rows.Scan(&pointID, &settlementID, &titleRu, &titleTr, &titleUa, &titleEn, &titleDe)
		if err != nil {
			return err
		}

		pointKey := "s" + strconv.Itoa(int(pointID.Int64))

		// FIXME: УБрать "магические" числа (lang_id)
		sf.addSuggest(titleRu.String, pointKey, "ru", 1, 1)
		sf.addSuggest(titleUa.String, pointKey, "uk", 1, 1)
		sf.addSuggest(titleTr.String, pointKey, "tr", 1, 1)
		sf.addSuggest(titleEn.String, pointKey, "en", 1, 1)
		sf.addSuggest(titleDe.String, pointKey, "de", 1, 1)
	}

	// Load settlements
	appLogger.Info("Load and index settlements")
	sqlTxt := "SELECT s.id, t.ru_nominative, t.tr_nominative, t.uk_nominative, t.en_nominative, t.de_nominative, s._geo_id FROM www_settlement as s JOIN www_translated_title AS t ON new_L_title_id = t.id WHERE s.majority_id < 5 or s.majority_id = 99"
	rows, err = con.Query(sqlTxt)
	if err != nil {
		return err
	}

	for rows.Next() {
		var (
			pointID       sql.NullInt64
			geoID         sql.NullInt64
			titleRu       sql.NullString
			titleUa       sql.NullString
			titleUaString string
			titleTr       sql.NullString
			titleEn       sql.NullString
			titleDe       sql.NullString
		)

		err = rows.Scan(&pointID, &titleRu, &titleTr, &titleUa, &titleEn, &titleDe, &geoID)
		if err != nil {
			return err
		}

		pointKey := "c" + strconv.Itoa(int(pointID.Int64))

		// Заполним из геобазы, если в СУБД пусто
		if titleUa.String == "" {
			titleUaString, err = GeobaseTitle(titleUa.String, int(geoID.Int64), "uk")
			if err != nil {
				return err
			}
		} else {
			titleUaString = titleUa.String
		}

		// FIXME: УБрать "магические" числа (lang_id, order_id)

		sf.addSuggest(titleRu.String, pointKey, "ru", 0, 1)
		sf.addSuggest(titleUaString, pointKey, "uk", 0, 1)
		sf.addSuggest(titleTr.String, pointKey, "tr", 0, 1)
		sf.addSuggest(titleEn.String, pointKey, "en", 0, 1)
		sf.addSuggest(titleDe.String, pointKey, "de", 0, 1)
	}

	appLogger.Info("Load and index countries")

	rows, err = con.Query("SELECT id, COALESCE(_geo_id, 0), COALESCE(title_ru, ''), COALESCE(title_uk, ''), COALESCE(title_tr, ''), COALESCE(title_en, '') FROM www_country")
	if err != nil {
		return err
	}

	for rows.Next() {
		var pointID int
		var geoID int
		var titleRu string
		var titleUa string
		var titleTr string
		var titleEn string

		err = rows.Scan(&pointID, &geoID, &titleRu, &titleUa, &titleTr, &titleEn)
		if err != nil {
			return err
		}

		pointKey := "l" + strconv.Itoa(pointID)

		// FIXME: УБрать "магические" числа (lang_id)
		sf.addSuggest(titleRu, pointKey, "ru", 2, 2)
		sf.addSuggest(titleUa, pointKey, "uk", 2, 2)
		sf.addSuggest(titleTr, pointKey, "tr", 2, 2)
		sf.addSuggest(titleEn, pointKey, "en", 2, 2)

		// Надо забрать еще синонимы
		synonyms, err := GetGeoBaseSynonyms(geoID)
		if err != nil {
			return err
		}

		for _, title := range strings.Split(synonyms, ",") {
			title = strings.Trim(title, " ")
			sf.addSuggest(title, pointKey, "synonym", 2, 2)
		}
	}

	appLogger.Info("Load and index codes from code system")
	// Берем только Сирену и ИАТА
	if takeHiddenAirports {
		query = "SELECT s.id, sc.code, sc.system_id FROM www_station s JOIN www_stationcode sc ON s.id = sc.station_id WHERE sc.system_id IN (1, 4) and s.majority_id <= ?"
	} else {
		query = "SELECT s.id, sc.code, sc.system_id FROM www_station s JOIN www_stationcode sc ON s.id = sc.station_id WHERE sc.system_id IN (1, 4) AND s.hidden = 0 and s.majority_id <= ?"
	}
	rows, err = con.Query(query, MaxStationMajority)
	if err != nil {
		return err
	}

	for rows.Next() {
		var pointID int
		var code string
		var systemID int

		err = rows.Scan(&pointID, &code, &systemID)
		if err != nil {
			return err
		}

		pointKey := "s" + strconv.Itoa(pointID)

		// Не надо добавлять код, если pointKey отсуствует в справочнике
		// Например, аэропорт скрыт
		if _, ok := sf.PointMap[pointKey]; ok {
			sf.addSuggest(code, pointKey, "code", 1, 2)
		}

	}

	appLogger.Info("Load and index codes from code settlements")
	rows, err = con.Query("SELECT id, COALESCE(iata, '') FROM www_settlement")
	if err != nil {
		return err
	}

	for rows.Next() {
		var pointID int
		var code string

		err = rows.Scan(&pointID, &code)
		if err != nil {
			return err
		}

		pointKey := "c" + strconv.Itoa(pointID)

		if _, ok := sf.PointMap[pointKey]; ok {
			sf.addSuggest(code, pointKey, "code", 0, 2)
		}
	}

	err = loadSynonyms(sf, con)
	if err != nil {
		return err
	}

	// Удалим дубликаты
	appLogger.Info("Remove duplicates from indexes")
	for lang := range sf.NGramIndexes {
		for k, v := range sf.NGramIndexes[lang].CharsBeginsMap {
			sf.NGramIndexes[lang].CharsBeginsMap[k] = RemoveDuplicates(v)
		}

		for k, v := range sf.NGramIndexes[lang].DifficultCharsMap {
			sf.NGramIndexes[lang].DifficultCharsMap[k] = RemoveDuplicates(v)
		}

		for k, v := range sf.NGramIndexes[lang].DifficultCharsMap {
			sf.NGramIndexes[lang].CharsOthersMapAll[k] = RemoveDuplicates(v)
		}
	}
	return nil
}

func loadSynonyms(sf *SuggestFabric, con *sql.DB) error {
	appLogger.Info("Load and index synonyms")
	rows, err := con.Query(`
			SELECT title, object_id, COALESCE(language, '')
			FROM www_pointsynonym WHERE content_type_id=14
		`)
	if err != nil {
		return err
	}

	synonymLangs := map[string]struct{}{
		"ru": {},
		"uk": {},
		"tr": {},
		"en": {},
		"de": {},
	}

	for rows.Next() {
		var title string
		var objectID int
		var lang string

		err = rows.Scan(&title, &objectID, &lang)
		if err != nil {
			return err
		}

		pointKey := "c" + strconv.Itoa(objectID)

		if _, ok := sf.PointMap[pointKey]; !ok {
			continue
		}
		if _, ok := synonymLangs[lang]; ok {
			sf.addSuggest(title, pointKey, lang, 0, 1)
			continue
		}
		sf.addSuggest(title, pointKey, "synonym", 0, 1)
	}
	return nil
}

// Догрузим и проиндексируем station2settlement
func LoadSettlement2Station(con *sql.DB, stationMap map[string]string) (map[string]string, map[string][]string, error) {
	m := make(map[string]string)
	station2settlement := make(map[string][]string)

	appLogger.Info("Load station2settlement")
	rows, err := con.Query("SELECT station_id, settlement_id FROM www_station2settlement")
	if err != nil {
		return nil, nil, err
	}

	for rows.Next() {
		var stationID int
		var settlementID int

		err = rows.Scan(&stationID, &settlementID)
		if err != nil {
			return nil, nil, err
		}

		settlementKey := "c" + strconv.Itoa(settlementID)
		stationKey := "s" + strconv.Itoa(stationID)
		if _, ok := stationMap[stationKey]; ok {
			m[settlementKey] = stationKey
		}

		station2settlement[stationKey] = append(station2settlement[stationKey], settlementKey)
	}

	return m, station2settlement, nil
}

func LoadGeoID(con *sql.DB) (map[int]string, error) {
	m := make(map[int]string)

	appLogger.Info("Load GeoID")
	appLogger.Info("Settlements")
	rows, err := con.Query("select id, COALESCE(_geo_id, 0) from www_settlement where _geo_id IS NOT NULL")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var pointID int
		var geoID int

		err = rows.Scan(&pointID, &geoID)
		if err != nil {
			return nil, err
		}

		m[geoID] = "c" + strconv.Itoa(pointID)
	}

	appLogger.Info("Countries")
	rows, err = con.Query("select id, COALESCE(_geo_id, 0) from www_country where _geo_id IS NOT NULL")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var pointID int
		var geoID int

		err = rows.Scan(&pointID, &geoID)
		if err != nil {
			return nil, err
		}

		m[geoID] = "l" + strconv.Itoa(pointID)
	}

	return m, nil
}

func LoadNationalVersions(con *sql.DB) (map[string]int, error) {
	query := `
		SELECT
			id, code
		FROM
			avia_nationalversion
	`

	appLogger.Info("Load national versions")
	rows, err := con.Query(query)
	if err != nil {
		return nil, err
	}

	m := make(map[string]int)

	var id int
	var code string
	for rows.Next() {
		err = rows.Scan(&id, &code)
		if err != nil {
			return nil, err
		}
		m[code] = id
	}

	return m, nil
}

func LoadDisputedPoints(con *sql.DB) (map[string]int, error) {
	m := make(map[string]int)
	nationalVersions := [3]string{"ru", "ua", "tr"}

	appLogger.Info("Load disputed territories")
	rows, err := con.Query("select id, COALESCE(_geo_id, 0) from www_settlement where disputed_territory = 1")
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		var settlementID int
		var geoID int

		err = rows.Scan(&settlementID, &geoID)
		if err != nil {
			return nil, err
		}

		if geoID <= 0 {
			continue
		}

		settlementKey := "c" + strconv.Itoa(settlementID)

		if !config.Features.UseGeobase {
			continue
		}

		for _, nationalVersion := range nationalVersions {
			geoBaseURL := fmt.Sprintf("http://geobase.qloud.yandex.ru/v0/find_country/%v/?crimea_status=%v", geoID, nationalVersion)
			body, err := GetHTTPBodyWithRetry(geoBaseURL, 5)
			if err != nil {
				return nil, fmt.Errorf("can't connect to geobase: %w", err)
			}
			countryID, err := strconv.Atoi(body)
			if err != nil {
				continue
			}

			key := nationalVersion + "_" + settlementKey
			m[key] = countryID

			// Добавим станции:
			sqlStr := "SELECT id FROM www_station WHERE settlement_id=? AND t_type_id=2 and majority_id <= ?"
			stRows, stErr := con.Query(sqlStr, settlementID, MaxStationMajority)
			if stErr != nil {
				return nil, stErr
			}
			for stRows.Next() {
				var stationID int

				err = stRows.Scan(&stationID)
				if err != nil {
					return nil, err
				}

				stationPointKey := "s" + strconv.Itoa(stationID)
				stationKey := nationalVersion + "_" + stationPointKey
				m[stationKey] = countryID
			}
		}
	}
	return m, nil
}

type GeoBaseLinguistic struct {
	NominativeCase string `json:"nominative_case"`
}

type GeoBaseSynonyms struct {
	Synonyms string `json:"synonyms"`
}

type errorString struct {
	s string
}

func (e *errorString) Error() string {
	return e.s
}

func GetHTTPBody(url string) (string, error) {
	var (
		req *http.Request
		err error
	)

	req, err = http.NewRequest("GET", url, nil)

	if err != nil {
		return "", err
	}

	req.Header.Set("Connection", "close")
	req.Header.Set("User-Agent", "avia suggests")

	resp, err := Client.Do(req)

	if err != nil {
		return "", err
	}
	defer resp.Body.Close()

	if resp.StatusCode != 200 {
		errMsg := fmt.Sprintf("Bad status code: %d %s", resp.StatusCode, url)
		return "", &errorString{errMsg}
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return "", err
	}

	return string(body), nil
}

func GetHTTPBodyWithRetry(url string, maxCount int) (string, error) {
	var (
		body string
		err  error
	)

	sleepTime := 100 * time.Millisecond

	for x := 0; x < maxCount; x++ {
		body, err = GetHTTPBody(url)

		if err == nil {
			break
		}
		time.Sleep(sleepTime)
	}

	return body, err
}

func GetGeoBaseSynonyms(geoID int) (string, error) {
	if geoID <= 0 {
		return "", nil
	}

	if !config.Features.UseGeobase {
		return "", nil
	}

	geoBaseURL := fmt.Sprintf("http://geobase.qloud.yandex.ru/v0/region_by_id/%v", geoID)
	body, err := GetHTTPBodyWithRetry(geoBaseURL, 5)
	if err != nil {
		return "", fmt.Errorf("can't connect to geobase: %w", err)
	}

	data := []byte(body)
	gbs := GeoBaseSynonyms{}
	err = json.Unmarshal(data, &gbs)
	if err == nil {
		return gbs.Synonyms, nil
	}

	return "", nil
}

func GeobaseTitle(title string, geoID int, lang string) (string, error) {
	if geoID <= 0 {
		return "", nil
	}

	if !config.Features.UseGeobase {
		return "", nil
	}

	geoBaseURL := fmt.Sprintf("http://geobase.qloud.yandex.ru/v0/linguistics_for_region/%v?lang=%v", geoID, lang)
	body, err := GetHTTPBodyWithRetry(geoBaseURL, 5)
	if err != nil {
		return "", fmt.Errorf("can't connect to geobase: %w", err)
	}

	data := []byte(body)
	gbl := GeoBaseLinguistic{}
	err = json.Unmarshal(data, &gbl)
	if err == nil {
		return gbl.NominativeCase, nil
	}

	return "", nil
}

func RemoveDuplicates(elements []SuggestMetaInfo) []SuggestMetaInfo {
	encountered := map[int64]bool{}
	var result []SuggestMetaInfo

	for v := range elements {
		if _, ok := encountered[elements[v].id]; !ok {
			encountered[elements[v].id] = true
			result = append(result, elements[v])
		}
	}

	return result
}

func reLoadData(successSleep time.Duration, failSleep time.Duration) {
	for {
		timeToSleep := successSleep
		newFabric := SuggestFabric{}
		err := newFabric.LoadAllData()
		if err == nil {
			sf = newFabric
			sf.SuccessLoaded = true

			appLogger.Info("Clear LRU cache")
			SuggestLRUCache.Clear()
		} else {
			appLogger.Error(err.Error())
			timeToSleep = failSleep
		}

		appLogger.Infof("Sleep %v", timeToSleep)
		time.Sleep(timeToSleep)
	}
}
