package main

import (
	"fmt"
	"net"
	"net/http"
	"regexp"
	"runtime/debug"
	"time"

	"github.com/karlseguin/ccache/v2"
	"github.com/labstack/echo/v4"

	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/avia/suggests/api/logging"
	"a.yandex-team.ru/travel/avia/suggests/api/yt"
)

var (
	// Фабрика данных
	sf SuggestFabric

	// Настройки
	config Config

	// LRU
	SuggestLRUCache = ccache.New(ccache.Configure().MaxSize(config.Engine.CacheSize).ItemsToPrune(uint32(config.Engine.CacheSize / 100)))

	// Internal stat
	SuggestStat EngineStat

	// Регексп для очистки входных данных
	cleanQueryStringParam        *regexp.Regexp
	cleanJSONPCallbackNameRegexp *regexp.Regexp

	originRegexp *regexp.Regexp

	// Http клиент
	Client *http.Client

	appLogger *arczap.Logger
	ytLogger  *yt.Logger
)

// Точка входа
func main() {
	// Только letters, digits, землю, дефис и пробел + все символы на буквах клавиатуры
	cleanQueryStringParam = regexp.MustCompile("[^\\p{L}\\d\\_\\-\\ \\{\\}\\;\\'\\,\\.\\[\\]\\<\\>\\:\\\"\\(\\)]+")

	// Требования к именам колбека
	cleanJSONPCallbackNameRegexp = regexp.MustCompile("^(callback|jQuery|suggest2_provider_jsonp_)([0-9])*$")

	// CORS
	originRegexp = regexp.MustCompile(`\.yandex\.(com|ru|ua|uz|com.tr|kz)$`)

	// Прочитаем настройки
	c, err := ReadConfig("config.yaml")
	if err != nil {
		panic(err)
	}
	config = *c

	// Подготовим клиент
	Client = makeHTTPClient()

	// logger set output to filelog
	appLogger, err = logging.NewApplicationLogger(&config.Logging)
	if err != nil {
		panic(err)
	}
	ytLogger, err = yt.NewLogger(config.Logging.YTLogPath)
	if err != nil {
		panic(err)
	}
	defer ytLogger.Close()

	// Increase GOGC
	debug.SetGCPercent(config.Engine.CGSize)

	// Setup CPU usage
	maxThreads := maxprocs.AdjustAuto()
	appLogger.Infof("Set GOMAXPROCS=%v", maxThreads)

	// Запустим регулярное обновление
	go reLoadData(time.Minute*60*3, time.Minute)

	SuggestStat = EngineStat{}

	// Echo instance
	e := echo.New()

	// Router
	e.GET("/avia", IndexV1)
	e.GET("/ttype/plane", IndexV1)
	e.GET("/v2/avia", IndexV2)
	e.GET("/v2/ttype/plane", IndexV2)
	e.GET("/stat/cache", CacheStat)

	probeService := probes.NewState(
		appLogger,
		probes.OnReady(
			func() error {
				if !sf.SuccessLoaded {
					return fmt.Errorf("indexes not loaded")
				}
				return nil
			},
		),
	)
	probes.EchoBind(config.Probe, e, probeService)

	// Start server
	hostWithPort := fmt.Sprintf("%s:%d", config.Listen.Host, config.Listen.Port)
	appLogger.Infof("Listen: %s", hostWithPort)
	go e.Logger.Fatal(e.Start(hostWithPort))
}

func makeHTTPClient() *http.Client {
	transport := &http.Transport{
		Dial: (&net.Dialer{
			Timeout: 500 * time.Millisecond,
		}).Dial,
		TLSHandshakeTimeout: 500 * time.Millisecond,
	}

	return &http.Client{
		Timeout:   60 * time.Second,
		Transport: transport,
	}
}
