package main

import (
	"encoding/json"
	"fmt"
)

type PriceIndexItem struct {
	FromID            int `json:"from_id"`
	ToID              int `json:"to_id"`
	BaseValue         int `json:"base_value"`
	NationalVersionID int `json:"national_version_id"`
}

type PriceIndexReponse struct {
	Status string           `json:"status"`
	Data   []PriceIndexItem `json:"data"`
}

// from -> to -> price
type PriceStorage map[int]map[int]int

type PriceFetcher struct {
	// nv -> PriceStorage
	storage map[int]PriceStorage
	url     string
	ready   bool
}

func (priceFetcher *PriceFetcher) Load() error {
	appLogger.Info("Loading prices")

	response, err := GetHTTPBodyWithRetry(priceFetcher.url, 5)
	if err != nil {
		return fmt.Errorf("can not load prices from price index: %v", err)
	}

	newStorage, err := parsePriceIndexResponse([]byte(response))
	if err != nil {
		return fmt.Errorf("can not parse price index response: %v\n%v", err, response)
	}

	priceFetcher.storage = newStorage
	for nv, nvStorage := range priceFetcher.storage {
		appLogger.Infof("NV: %d, nubmber of records: %d", nv, len(nvStorage))
	}
	return nil
}

func parsePriceIndexResponse(response []byte) (map[int]PriceStorage, error) {
	var parsedResponse PriceIndexReponse
	err := json.Unmarshal(response, &parsedResponse)
	if err != nil {
		return nil, err
	}

	if parsedResponse.Status != "ok" {
		return nil, fmt.Errorf("status from price-index: %v", parsedResponse.Status)
	}

	storage := make(map[int]PriceStorage)
	for _, item := range parsedResponse.Data {
		nv := item.NationalVersionID
		nvStorage, ok := storage[nv]
		if !ok {
			nvStorage = PriceStorage{}
		}

		_, ok = nvStorage[item.FromID]
		if !ok {
			nvStorage[item.FromID] = make(map[int]int)
		}

		nvStorage[item.FromID][item.ToID] = item.BaseValue
		storage[nv] = nvStorage
	}

	return storage, nil
}

func (priceFetcher *PriceFetcher) GetPrice(nationalVersion int, fromID int, toID int) int {
	nvStorage, ok := priceFetcher.storage[nationalVersion]
	if !ok {
		return -1
	}
	fromIDStorage, ok := nvStorage[fromID]
	if !ok {
		return -1
	}

	result, ok := fromIDStorage[toID]
	if !ok {
		return -1
	}

	return result
}
