package main

import (
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/travel/avia/suggests/api/yt"
)

// Пользовательский запрос в виде объекта
type Query struct {
	rawQuery             string
	nationalVersion      string
	lang                 string
	langID               int
	otherPoint           string
	otherQuery           string
	callbackName         string
	suggestLen           int
	needCountry          bool
	needStation          bool
	field                string
	showHiddenField      bool // Возвращать ли метку о том, что объект скрытый
	showHaveAirportField bool // Возвращать ли метку о том, что город имеет аэропорт
	blended              bool // Смешанное отображение
	maxCount             int  // Максимальное количество саджестов в ответе
	includePrices        bool // добавить цены в ответ
	geobaseID            int  // Идентификатор точки, используется вместо query
	showEqualAirports    bool // Show airports which name coincides with city name. Has effect only with blended == true
	flattenSuggests      bool // Игнорировать вложенность элементов друг в друга при формировании ответа
}

// Ключ для кеширования
func (q *Query) Key() string {
	parts := []string{
		q.otherPoint,
		q.otherQuery,
		q.rawQuery,
		q.nationalVersion,
		q.lang,
		strconv.FormatBool(q.needCountry),
		strconv.FormatBool(q.showHiddenField),
		strconv.FormatBool(q.showHaveAirportField),
		q.field,
		strconv.FormatBool(q.blended),
		strconv.Itoa(q.maxCount),
		strconv.FormatBool(q.includePrices),
		strconv.FormatBool(q.showEqualAirports),
	}

	return strings.Join(parts, "_")
}

func (q *Query) Log(cached string, d time.Duration, count int) {
	execTime := float32(d.Nanoseconds()) / 1e6
	appLogger.Infof(
		"Query: cache=%s, execution_time=%6.3f ms, rawQuery=%s, nationalVersion=%s, lang=%s, otherPoint=%s, otherQuery=%s, unixtime=%d, count=%d, maxCount=%d",
		cached,
		execTime,
		q.rawQuery,
		q.nationalVersion,
		q.lang,
		q.otherPoint,
		q.otherQuery,
		time.Now().Unix(),
		count,
		q.maxCount,
	)

	ytLogger.Log(
		yt.Record{
			NationalVersion: q.nationalVersion,
			Lang:            q.lang,
			Field:           q.field,
			NeedCountry:     q.needCountry,
			ExecutionTime:   execTime,
			FromCache:       cached,
			Count:           count,
			OtherPoint:      q.otherPoint,
			OtherQuery:      q.otherQuery,
			Unixtime:        time.Now().Unix(),
			RawQuery:        q.rawQuery,
			MaxCount:        q.maxCount,
		},
	)
}

func cleanJSONPCallbackName(callbackName string) string {
	if !cleanJSONPCallbackNameRegexp.MatchString(callbackName) {
		return ""
	}
	callbackRunes := []rune(callbackName)
	callbackRunesLen := len(callbackRunes)

	// Ограничим длинну
	if callbackRunesLen > 64 {
		callbackRunesLen = 64
	}

	return string(callbackRunes[:callbackRunesLen])
}

func cleanRawQuery(rawQuery string) string {
	rawQuery = cleanQueryStringParam.ReplaceAllString(strings.TrimSpace(rawQuery), "")

	queryRunes := []rune(rawQuery)
	queryRunesLen := len(queryRunes)

	// Ограничим длинну
	if queryRunesLen > 128 {
		return string(queryRunes[:128])
	} else {
		return rawQuery
	}
}

// Сформируем Query из пользовательского запроса
func (q *Query) FillFromRequest(ctx echo.Context, blended bool) {
	lang := strings.ToLower(ctx.FormValue("lang"))

	rawQuery := strings.ToLower(ctx.FormValue("query"))
	if rawQuery == "" {
		rawQuery = strings.ToLower(ctx.FormValue("part"))
	}

	rawQuery = cleanRawQuery(rawQuery)
	q.rawQuery = strings.Replace(rawQuery, "ё", "е", -1)

	q.nationalVersion = strings.ToLower(ctx.FormValue("national_version"))
	q.lang = lang
	q.langID = sf.LangMap[lang]
	q.callbackName = ctx.FormValue("callback")
	q.otherPoint = ctx.FormValue("other_point")
	q.otherQuery = ctx.FormValue("other_query")
	q.suggestLen = len([]rune(strings.ToLower(q.rawQuery)))
	q.showHiddenField, _ = strconv.ParseBool(ctx.FormValue("hidden_field"))
	q.showHaveAirportField, _ = strconv.ParseBool(ctx.FormValue("have_airport_field"))

	rawGeobaseID := ctx.FormValue("geobaseId")
	if rawGeobaseID == "" {
		q.geobaseID = 0
	} else {
		parsedGeobaseID, err := strconv.Atoi(rawGeobaseID)
		if err == nil {
			q.geobaseID = parsedGeobaseID
		} else {
			parsedGeobaseID = 0
		}
	}

	if ctx.FormValue("only_cities") == "true" {
		q.flattenSuggests = true
		q.needCountry = false
		q.needStation = false
	} else {
		q.flattenSuggests = ctx.FormValue("flatten_suggests") == "true"
		q.needCountry = ctx.FormValue("need_country") == "true"
		q.needStation = true
	}

	q.field = ctx.FormValue("field")
	q.includePrices = ctx.FormValue("includePrices") == "true"

	// Почистим то, что потом попадет на выдачу
	q.callbackName = cleanJSONPCallbackName(q.callbackName)
	q.blended = blended
	q.showEqualAirports = ctx.FormValue("showEqualAirports") == "true"
	maxCount, err := strconv.Atoi(ctx.FormValue("count"))
	if err == nil {
		q.maxCount = min(maxCount, config.Engine.MaxSuggest)
	} else {
		q.maxCount = config.Engine.DefaultSuggests
	}
}
