package main

import (
	"encoding/json"
	"fmt"
	"reflect"
	"sort"
	"strconv"
	"strings"
	"sync/atomic"
	"unsafe"

	"golang.org/x/sync/errgroup"
)

type NGramIndex struct {
	CharsBeginsMap    map[string][]SuggestMetaInfo // Индекс первых букв
	DifficultCharsMap map[string][]SuggestMetaInfo // Индекс букв для сложных случаев: Нью-Йорк, Баден-бкден
	CharsOthersMapAll map[string][]SuggestMetaInfo // Индекс остальных букв для всех слов
	TitlePointsMap    map[string][]SuggestMetaInfo // Title к саджесту
	Titles            [][]rune                     // Все title
	TitlesLenIndex    []int                        // Разбивка Titles по длинам

	AllInputs         [][]rune // Все возможные входы, для которых показываем саджесты
	AllInputsLenIndex []int    // Разбивка входов по длинам
}

func (ngramIndex *NGramIndex) Init() {
	ngramIndex.CharsBeginsMap = make(map[string][]SuggestMetaInfo)
	ngramIndex.DifficultCharsMap = make(map[string][]SuggestMetaInfo)
	ngramIndex.CharsOthersMapAll = make(map[string][]SuggestMetaInfo)

	ngramIndex.TitlePointsMap = make(map[string][]SuggestMetaInfo)
	ngramIndex.Titles = make([][]rune, 0)
	ngramIndex.TitlesLenIndex = make([]int, 0)

	ngramIndex.AllInputs = make([][]rune, 0)
	ngramIndex.AllInputsLenIndex = make([]int, 0)
}

func (ngramIndex *NGramIndex) CheckSuggestInBeginIndex(chars string, sm SuggestMetaInfo) bool {
	for _, suggest := range ngramIndex.CharsBeginsMap[chars] {
		if suggest.pointKey == sm.pointKey {
			return true
		}
	}
	return false
}

func (ngramIndex *NGramIndex) BuildTitleIndex() {
	sort.Sort(StringByLens(ngramIndex.Titles))
	ngramIndex.TitlesLenIndex = buildLenIndex(ngramIndex.Titles)

	for input := range ngramIndex.CharsBeginsMap {
		ngramIndex.AllInputs = append(ngramIndex.AllInputs, []rune(input))
	}

	sort.Sort(StringByLens(ngramIndex.AllInputs))
	ngramIndex.AllInputsLenIndex = buildLenIndex(ngramIndex.AllInputs)
}

// Автономная фабрика обрабатывающая запрос
type SuggestFabric struct {
	SuggestID              int64
	Point2IATAMap          map[string]string            // IATA
	Point2ICAOMap          map[string]string            // ICAO
	Point2SirenaMap        map[string]string            // Sirena
	PopularDirectionsMap   map[string]int               // Популярные направления
	Top15DirectionsMap     map[string][]SuggestMetaInfo // TOP15 для дефолтных саджестов
	Top15SettlementsMap    map[string][]SuggestMetaInfo // TOP 15 городов нац. версии
	PointMap               map[string]Point             // Points
	RegionMap              map[int]Region               // Регионы
	CountryMap             map[int]Country              // Страны
	NGramIndexes           map[int]NGramIndex           // Индексы по языкам
	LangMap                map[string]int               // Мап с языками
	ReverseLangMap         map[int]string               // Из кода в язык
	PopularityMap          map[string]int               // Временный мап с популярностью
	PopularCities          map[string][]SuggestMetaInfo // Популряные города стран
	MajorityMap            map[string]int               // Значимость городов/аэропортов
	StationMap             map[string]string            // Связь станций городов
	Settlement2StationMap  map[string]string            // Settlement2Station
	Station2SettlementMap  map[string][]string          // Station2Settlement
	Settlement2AllStations map[string][]string          // Все станции города
	SuccessLoaded          bool                         // Флаг удачной загрузки
	DisputedTerritories    map[string]int               // Спорные территории
	GeoID2Point            map[int]string               // GeoId к точкам (городам и странам)
	PriceStorage           PriceFetcher                 // хранилизе цен
	NationalVersionToID    map[string]int
}

// Для стран отдаем ISO код
func (sf *SuggestFabric) getCode(pointKey string) (string, bool) {
	if getPointType(pointKey) == CountryType {
		numberKey, _ := strconv.Atoi(pointKey[1:])
		country, ok := sf.CountryMap[numberKey]
		if ok {
			return country.code, ok
		} else {
			return "", false
		}
	}

	pointCode, ok := sf.Point2IATAMap[pointKey]

	if !ok {
		pointCode, ok = sf.Point2SirenaMap[pointKey]
	}

	return pointCode, ok
}

func fillPuntedField(suggests []SuggestMetaInfo, value bool) {
	for i := 0; i < len(suggests); i++ {
		suggests[i].punted = value
	}
}

func (sf *SuggestFabric) InitSettlement2AllStations() {
	sf.Settlement2AllStations = make(map[string][]string)
	for stationKey, settlementKey := range sf.StationMap {
		stations, ok := sf.Settlement2AllStations[settlementKey]
		if ok {
			sf.Settlement2AllStations[settlementKey] = append(stations, stationKey)
		} else {
			sf.Settlement2AllStations[settlementKey] = []string{stationKey}
		}
	}

	for settlementKey, stationKey := range sf.Settlement2StationMap {
		stations, ok := sf.Settlement2AllStations[settlementKey]
		if ok {
			sf.Settlement2AllStations[settlementKey] = append(stations, stationKey)
		} else {
			sf.Settlement2AllStations[settlementKey] = []string{stationKey}
		}
	}
}

func (sf *SuggestFabric) FindCode(pointKey string) string {
	code, ok := sf.getCode(pointKey)

	if !ok {
		pointKey := sf.Settlement2StationMap[pointKey]
		code, _ = sf.getCode(pointKey)
	}

	return code
}

func boolToString(value bool) string {
	if value {
		return "1"
	}

	return "0"
}

func addQuotes(s string) string {
	return fmt.Sprintf("\"%s\"", s)
}

func (sf *SuggestFabric) getPointDescription(s SuggestMetaInfo, q Query) string {
	pointKey := s.pointKey
	p := sf.PointMap[pointKey]

	disputedKey := q.nationalVersion + "_" + pointKey
	countryID, ok := sf.DisputedTerritories[disputedKey]

	if !ok {
		countryID = p.countryID
	}
	countryTitle := sf.CountryMap[countryID].Title(q.lang)
	regionTitle := sf.RegionMap[p.regionID].Title(q.lang)

	var cityTitle string
	switch getPointType(pointKey) {
	case StationType:
		cityTitle = sf.PointMap[sf.StationMap[pointKey]].Title(q.lang)
	case SettlementType:
		cityTitle = sf.PointMap[pointKey].Title(q.lang)
	case CountryType:
		cityTitle = ""
	}

	pointCode := sf.FindCode(pointKey)

	var pointDescr string
	if q.blended {
		pointDescr = "{" + strings.Join([]string{
			"\"point_key\":" + addQuotes(pointKey),
			"\"point_code\":" + addQuotes(pointCode),
			"\"region_title\":" + addQuotes(regionTitle),
			"\"city_title\":" + addQuotes(cityTitle),
			"\"country_title\":" + addQuotes(countryTitle),
			"\"missprint\":" + boolToString(s.misprint),
			"\"hidden\":" + boolToString(p.hidden),
			"\"have_airport\":" + boolToString(p.haveAirport),
			"\"added\":" + boolToString(s.added),
			"\"have_not_hidden_airport\":" + boolToString(p.haveNotHiddenAirport),
		}, ", ") + "}"
	} else {
		// Общие поля: страна, регион, код, опечатки
		pointDescr = "{\"country\":\"" + countryTitle + "\",\"region\":\"" + regionTitle + "\",\"code\":\"" + pointCode + "\",\"id\":\"" + pointKey + "\", \"mp\":" + boolToString(s.misprint)

		if q.showHiddenField {
			pointDescr += ", \"hidden\":" + boolToString(p.hidden)
		}

		if q.showHaveAirportField {
			pointDescr += ", \"have_airport\":" + boolToString(p.haveAirport)
		}

		// Конец структуры
		pointDescr += "}"
	}

	return pointDescr
}

func (sf *SuggestFabric) getSettlementCode(code string) int {
	pointType := getPointType(code)
	if pointType == CountryType {
		return -1
	}

	pointCode, _ := strconv.Atoi(code[1:])
	if pointType == SettlementType {
		return pointCode
	}

	settlementCode, ok := sf.StationMap[code]
	if ok {
		pointCode, _ = strconv.Atoi(settlementCode[1:])
		return pointCode
	}

	return -1
}

func (sf *SuggestFabric) findPoints(s SuggestMetaInfo, q Query) (int, int) {
	fromCode, toCode := s.pointKey, q.otherPoint
	if q.otherPoint == "" {
		return -1, -1
	}
	if q.field == "to" {
		toCode, fromCode = fromCode, toCode
	}

	return sf.getSettlementCode(fromCode), sf.getSettlementCode(toCode)
}

func (sf *SuggestFabric) MakeResponseForSuggest(ind int, suggests []SuggestMetaInfo, q Query) string {
	s := suggests[ind]
	p := sf.PointMap[s.pointKey]
	var pointType string
	if q.blended {
		pointType = strconv.Itoa(getPointType(s.pointKey))
	} else {
		if strings.HasPrefix(s.pointKey, "l") {
			pointType = "country"
		} else if strings.HasPrefix(s.pointKey, "c") {
			pointType = "city"
		} else if strings.HasPrefix(s.pointKey, "s") {
			pointType = "airport"
		}

		pointType = addQuotes(pointType)
	}

	pointDescription := sf.getPointDescription(s, q)

	priceStr := "null"
	fromID, toID := sf.findPoints(s, q)
	if (fromID != -1) && (toID != -1) {
		nv, ok := sf.NationalVersionToID[q.nationalVersion]
		price := -1
		if ok {
			price = sf.PriceStorage.GetPrice(nv, fromID, toID)
		}
		if price != -1 {
			priceStr = strconv.FormatInt(int64(price), 10)
		}
	}

	suggestString := "[" + pointType + ",\"" + p.Title(q.lang) + "\", " + pointDescription
	if q.blended {
		suggestString += ", ["
		if ind != len(suggests)-1 {
			currentLevel := s.level
			additionalStrings := make([]string, 0)
			for j := ind + 1; j < len(suggests); j++ {
				if suggests[j].level == currentLevel+1 {
					additionalStrings = append(
						additionalStrings,
						sf.MakeResponseForSuggest(j, suggests, q),
					)
				}

				if suggests[j].level == currentLevel {
					break
				}
			}
			suggestString += strings.Join(additionalStrings, ", ")
		}
		suggestString += "]"
	}
	if q.includePrices {
		suggestString += "," + priceStr
	}
	suggestString += "]"
	return suggestString
}

// Сделаем JSON как string ()
func (sf *SuggestFabric) MakeResponse(suggests []SuggestMetaInfo, q Query) string {
	var suggestsStrings []string

	for i := 0; i < len(suggests); i++ {
		s := suggests[i]
		if s.level == 0 {
			suggestsStrings = append(suggestsStrings, sf.MakeResponseForSuggest(i, suggests, q))
		}
	}

	var encodedQuery string
	if q.geobaseID != 0 {
		encodedQuery = strconv.Itoa(q.geobaseID)
	} else {
		queryInBytes, _ := json.Marshal(q.rawQuery)
		encodedQuery = string(queryInBytes)
	}

	response := "[" + encodedQuery + ",[" + strings.Join(suggestsStrings, ", ") + "]]"

	return response
}

// Отсортируем по город/аэропорт с сохранение существующего порядка
func (sf *SuggestFabric) FilterAndSortByPointType(suggests []SuggestMetaInfo, needCountry bool, needStation bool) []SuggestMetaInfo {
	var cities []SuggestMetaInfo
	var airports []SuggestMetaInfo
	var other []SuggestMetaInfo

	for _, s := range suggests {
		switch s.pointTypeOrder {
		case 0:
			cities = append(cities, s)
		case 1:
			airports = append(airports, s)
		default:
			other = append(other, s)

		}
	}

	if needStation {
		suggests = append(cities, airports...)
	} else {
		suggests = cities
	}

	if needCountry {
		suggests = append(suggests, other...)
	}

	return suggests
}

func (sf *SuggestFabric) FixLevelForChildren(suggests []SuggestMetaInfo, parentLevel int64, parentIndex int) {
	for i := parentIndex + 1; i < len(suggests) && suggests[i].level != parentLevel; i++ {
		suggests[i].level -= 1
	}
}

func (sf *SuggestFabric) FilterByPointType(suggests []SuggestMetaInfo, needCountry bool, needStation bool) []SuggestMetaInfo {
	var result []SuggestMetaInfo
	for i, s := range suggests {
		switch s.pointTypeOrder {
		case 0:
			result = append(result, s)
		case 1:
			if needStation {
				result = append(result, s)
			} else {
				sf.FixLevelForChildren(suggests, s.level, i)
			}
		default:
			if needCountry {
				result = append(result, s)
			} else {
				sf.FixLevelForChildren(suggests, s.level, i)
			}
		}
	}

	return result
}

type StringByLens [][]rune

func (lines StringByLens) Len() int           { return len(lines) }
func (lines StringByLens) Less(i, j int) bool { return len(lines[i]) < len(lines[j]) }
func (lines StringByLens) Swap(i, j int)      { lines[i], lines[j] = lines[j], lines[i] }

func buildLenIndex(lines [][]rune) []int {
	maxLen := len(lines[len(lines)-1])
	lenIndex := make([]int, maxLen)

	prevLen := 0

	for ind := 0; ind < len(lines); {
		currentLen := len(lines[ind])
		for ind < len(lines) && len(lines[ind]) == currentLen {
			ind++
		}
		lenIndex[currentLen-1] = ind
		// Заполняем пропущенные значения
		for helpInd := prevLen; helpInd != currentLen-1; helpInd++ {
			lenIndex[helpInd] = lenIndex[max(prevLen-1, 0)]
		}
		prevLen = currentLen
	}
	return lenIndex
}

func convertKeyIndMapToSlice(suggests []SuggestMetaInfo, pointKeyToInd *map[string]int) []SuggestMetaInfo {
	result := make([]SuggestMetaInfo, 0, len(*pointKeyToInd))
	for i, suggest := range suggests {
		if ind, ok := (*pointKeyToInd)[suggest.pointKey]; ok && ind == i {
			result = append(result, suggest)
		}
	}
	return result
}

func makePointKeyToInd(suggests []SuggestMetaInfo) map[string]int {
	pointKeyToInd := make(map[string]int)
	for i, suggest := range suggests {
		pointKey := suggest.pointKey
		prevInd, ok := pointKeyToInd[pointKey]
		if !ok || suggests[prevInd].level < suggest.level {
			pointKeyToInd[pointKey] = i
		}
	}
	return pointKeyToInd
}

func (sf *SuggestFabric) RemoveDuplicates(suggests []SuggestMetaInfo) []SuggestMetaInfo {
	pointKeyToInd := makePointKeyToInd(suggests)
	return convertKeyIndMapToSlice(suggests, &pointKeyToInd)
}

func (sf *SuggestFabric) RemoveAirportsEqualToCities(suggests []SuggestMetaInfo, lang string) []SuggestMetaInfo {
	pointKeyToInd := makePointKeyToInd(suggests)
	for i, suggest := range suggests {
		pointKey := suggest.pointKey
		if getPointType(pointKey) == SettlementType {
			for _, airportKey := range sf.Settlement2AllStations[pointKey] {
				if ind, ok := pointKeyToInd[airportKey]; ok && suggests[ind].level <= suggests[i].level {
					airportTitle := sf.PointMap[airportKey].Title(lang)
					settlementTitle := sf.PointMap[pointKey].Title(lang)
					if airportTitle == settlementTitle {
						delete(pointKeyToInd, airportKey)
					}
				}
			}
		}
	}
	return convertKeyIndMapToSlice(suggests, &pointKeyToInd)
}

// Почистим выдачу от одинаковых саджестов
func (sf *SuggestFabric) UniqSuggests(suggests []SuggestMetaInfo, limit int) []SuggestMetaInfo {
	encountered := map[string]bool{}
	var result []SuggestMetaInfo

	for s := 0; s < len(suggests); s++ {
		if _, ok := encountered[suggests[s].pointKey]; !ok {
			encountered[suggests[s].pointKey] = true
			result = append(result, suggests[s])

			// Горшочек, не вари!
			if len(result) == limit {
				break
			}
		}
	}

	return result
}

func withCapturePanics(f func() error) func() error {
	return func() (err error) {
		defer func() {
			if r := recover(); r != nil {
				err = fmt.Errorf("recovered LoadAllData: %v", r)
			}
		}()
		return f()
	}
}

func (sf *SuggestFabric) LoadAllData() (err error) {
	defer func() {
		if r := recover(); r != nil {
			err = fmt.Errorf("recovered LoadAllData: %v", r)
		}
	}()

	con, err := NewConnection()
	if err != nil {
		return err
	}
	defer con.Close()

	// Обнулим каунтеры
	sf.SuggestID = 0

	// Загрузим справочники нужные для работы остальных
	g := new(errgroup.Group)
	pointMapEvent := make(chan interface{})
	countryMapEvent := make(chan interface{})
	stationMapEvent := make(chan interface{})
	loaders := []func() error{
		func() error {
			var err error
			sf.PopularityMap, err = LoadPopularity(con)
			return err
		},
		func() error {
			var err error
			sf.MajorityMap, err = LoadMajority(con)
			return err
		},
		func() error {
			defer close(stationMapEvent)
			var err error
			sf.StationMap, err = LoadStations(con, config.Features.ShowHiddenAirports)
			return err
		},
		func() error {
			defer close(pointMapEvent)
			var err error
			sf.PointMap, err = LoadPoints(con)
			return err
		},
		func() error {
			defer close(countryMapEvent)
			var err error
			sf.CountryMap, err = LoadCountries(con)
			return err
		},
		func() error {
			var err error
			sf.Point2IATAMap, err = LoadIATA(con)
			return err
		},
		func() error {
			var err error
			sf.Point2SirenaMap, err = LoadSirena(con)
			return err
		},
		func() error {
			var err error
			sf.RegionMap, err = LoadRegions(con)
			return err
		},
		func() error {
			var err error
			sf.PopularDirectionsMap, err = LoadPopularDirections(con)
			return err
		},
		func() error {
			var err error
			<-pointMapEvent
			<-countryMapEvent
			sf.Top15DirectionsMap, err = LoadTop15(con, sf.PointMap, sf.CountryMap)
			return err
		},
		func() error {
			var err error
			sf.Top15SettlementsMap, err = loadTop15Settlements(con)
			return err
		},
		func() error {
			var err error
			sf.PopularCities, err = loadPopularSettlementsByCountries(con)
			return err
		},
		func() error {
			var err error
			<-stationMapEvent
			sf.Settlement2StationMap, sf.Station2SettlementMap, err = LoadSettlement2Station(con, sf.StationMap)
			return err
		},
		func() error {
			var err error
			sf.DisputedTerritories, err = LoadDisputedPoints(con)
			return err
		},
		func() error {
			var err error
			sf.GeoID2Point, err = LoadGeoID(con)
			return err
		},
	}
	for _, f := range loaders {
		g.Go(withCapturePanics(f))
	}
	err = g.Wait()
	if err != nil {
		return err
	}
	sf.InitSettlement2AllStations()

	sf.PriceStorage.url = config.Services.PriceIndexURL
	err = sf.PriceStorage.Load()
	if err != nil {
		return err
	}
	sf.NationalVersionToID, err = LoadNationalVersions(con)
	if err != nil {
		return err
	}

	// TODO: Вынести в отдельную фун-цию
	sf.LangMap = make(map[string]int)
	sf.LangMap["ru"] = 1
	sf.LangMap["uk"] = 2
	sf.LangMap["tr"] = 3
	sf.LangMap["en"] = 4
	sf.LangMap["de"] = 5
	sf.LangMap["synonym"] = 98
	sf.LangMap["code"] = 99

	sf.ReverseLangMap = make(map[int]string)
	for lang, langID := range sf.LangMap {
		sf.ReverseLangMap[langID] = lang
	}

	// проиндексируем саджесты
	sf.InitIndexes()
	sf.IndexSettlement2Station()

	// TODO: Может внутрь фабрики загнать функцию?
	err = LoadSuggests(sf, con, config.Features.ShowHiddenAirports)
	if err != nil {
		return err
	}

	for _, langID := range sf.LangMap {
		ngramIndex := sf.NGramIndexes[langID]
		ngramIndex.BuildTitleIndex()
		sf.NGramIndexes[langID] = ngramIndex
	}

	// Выведем статистику кол-ва элементов
	sf.PrintLoadStatistics()

	return nil
}

func (sf *SuggestFabric) InitIndexes() {
	sf.NGramIndexes = make(map[int]NGramIndex)
	for _, langID := range sf.LangMap {
		ngramIndex := NGramIndex{}
		ngramIndex.Init()
		sf.NGramIndexes[langID] = ngramIndex
	}
}

func (sf *SuggestFabric) IndexSettlement2Station() {
	appLogger.Info("Index settlement2station")
	for settlementKey := range sf.Settlement2StationMap {
		if point, ok := sf.PointMap[settlementKey]; ok {
			sf.addSuggest(point.Title("ru"), settlementKey, "ru", 0, 1)
			sf.addSuggest(point.Title("uk"), settlementKey, "uk", 0, 1)
			sf.addSuggest(point.Title("tr"), settlementKey, "tr", 0, 1)
			sf.addSuggest(point.Title("en"), settlementKey, "en", 0, 1)
		}
	}
}

// Проиндексируем строку по любым комбинациям рядом стоящих букв
func (sf *SuggestFabric) IndexRune(titleRunes []rune, sm SuggestMetaInfo) {
	titleLen := len(titleRunes)

	ngramIndex := sf.NGramIndexes[sm.langID]
	for indLen := 1; indLen <= titleLen; indLen++ {
		for indPos := 0; indPos <= titleLen-indLen; indPos++ {
			runes := titleRunes[indPos : indPos+indLen]
			chars := string(runes)

			if indPos == 0 {
				ngramIndex.CharsBeginsMap[chars] = append(ngramIndex.CharsBeginsMap[chars], sm)
			} else {
				// Добавим только если этот же саджест не добавили в CharsBeginsMap
				// Почему: Баден-баден, Факфак, Мумбаи, etc, etc
				alreadyInIndex := false
				for _, s := range ngramIndex.CharsBeginsMap[chars] {
					if s.id == sm.id {
						alreadyInIndex = true
						break
					}
				}

				if !alreadyInIndex {
					ngramIndex.CharsOthersMapAll[chars] = append(ngramIndex.CharsOthersMapAll[chars], sm)
				}
			}
		}
	}

	sf.NGramIndexes[sm.langID] = ngramIndex

	sf.IndexRuneForDifficultChars(titleRunes, sm)
}

func isDifficultRune(r rune) bool {
	return r == rune(' ') || r == rune('-')
}

func (sf *SuggestFabric) IndexRuneForDifficultChars(titleRunes []rune, sm SuggestMetaInfo) {
	titlesLen := len(titleRunes)
	ngramIndex := sf.NGramIndexes[sm.langID]
	for suggestStart := 0; suggestStart < titlesLen-2; suggestStart++ {
		if isDifficultRune(titleRunes[suggestStart]) {
			// Skip difficult character
			for suggestEnd := suggestStart + 2; suggestEnd <= titlesLen; suggestEnd++ {
				suggestBody := titleRunes[suggestStart:suggestEnd]
				chars := string(suggestBody)
				if !ngramIndex.CheckSuggestInBeginIndex(chars, sm) {
					ngramIndex.DifficultCharsMap[chars] = append(ngramIndex.DifficultCharsMap[chars], sm)
				}

				suggestBody = titleRunes[suggestStart+1 : suggestEnd]
				chars = string(suggestBody)
				if !ngramIndex.CheckSuggestInBeginIndex(chars, sm) {
					ngramIndex.DifficultCharsMap[chars] = append(ngramIndex.DifficultCharsMap[chars], sm)
				}
			}
		}
	}

}

func (sf *SuggestFabric) SortSuggests(suggests []SuggestMetaInfo, q Query) []SuggestMetaInfo {
	// Заполняем то, что не смогли в прекеше
	for i, s := range suggests {
		if q.langID == s.langID {
			suggests[i].sameLang = true
		} else {
			suggests[i].sameLang = false
		}

		if s.langID == sf.LangMap["synonym"] {
			suggests[i].isCommonSynonym = true
		} else {
			suggests[i].isCommonSynonym = false
		}

		if s.pointKey == q.otherPoint {
			suggests[i].likeOther = true
		} else {
			suggests[i].likeOther = false
		}

		// Приоритезируем IATA
		if q.suggestLen == s.suggestLen && (s.langID == 99 || s.langID == 98) && !s.punted {
			suggests[i].likeCode = true
		} else {
			suggests[i].likeCode = false
		}

		suggests[i].directionPopularity = 0
		otherPointKeys := []string{q.otherPoint}
		if getPointType(q.otherPoint) == StationType {
			otherPointKeys = []string{sf.StationMap[q.otherPoint]}
			if settlements, ok := sf.Station2SettlementMap[q.otherPoint]; ok {
				otherPointKeys = append(otherPointKeys, settlements...)
			}
		}

		pointKeys := []string{s.pointKey}
		if getPointType(q.otherPoint) == StationType {
			pointKeys = []string{sf.StationMap[s.pointKey]}
			if settlements, ok := sf.Station2SettlementMap[s.pointKey]; ok {
				pointKeys = append(pointKeys, settlements...)
			}
		}

		for _, otherPointKey := range otherPointKeys {
			for _, pointKey := range pointKeys {
				directionPopularityKey := q.nationalVersion + "_" + otherPointKey + "_" + pointKey
				directionPopularity, ok := sf.PopularDirectionsMap[directionPopularityKey]

				if ok {
					suggests[i].directionPopularity = max(directionPopularity, suggests[i].directionPopularity)
				}
			}
		}

		popularityKey := q.nationalVersion + "_" + s.pointKey
		popularity, ok := sf.PopularityMap[popularityKey]

		if ok {
			suggests[i].popularity = popularity
		} else {
			suggests[i].popularity = 0
		}

		if suggests[i].langID == sf.LangMap["code"] {
			suggests[i].fullSuggest = false
		} else {
			point := sf.PointMap[suggests[i].pointKey]
			var title string
			if suggests[i].langID != sf.LangMap["synonym"] {
				lang := sf.ReverseLangMap[suggests[i].langID]
				title = strings.ToLower(point.Title(lang))
			} else {
				title = strings.ToLower(suggests[i].title)
			}
			rawQuery := strings.ToLower(strings.Replace(q.rawQuery, "ё", "e", -1))
			suggests[i].fullSuggest = title == rawQuery
		}
	}

	if q.blended {
		sort.Sort(SuggestMetaInfoSortBlended(suggests))
	} else {
		sort.Sort(SuggestMetaInfoSort(suggests))
	}

	return suggests
}

func split(s string, sepators []rune) []string {
	f := func(r rune) bool {
		for _, sep := range sepators {
			if r == sep {
				return true
			}
		}
		return false
	}

	fields := strings.FieldsFunc(s, f)
	resFields := make([]string, 0)
	for _, field := range fields {
		if len(field) > 0 {
			resFields = append(resFields, field)
		}
	}

	return resFields
}

func lessPointTypes(pointType, otherPointType rune) bool {
	if pointType == 's' { // Airport is minimal
		return otherPointType != 's'
	}

	if pointType == 'c' {
		return otherPointType == 'l'
	}

	return false
}

func (sf *SuggestFabric) isCommon(pointKey1, pointKey2 string) bool {
	if pointKey1 == pointKey2 {
		return true
	}

	if pointKey1[0] == pointKey2[0] { // Same types but different codes
		return false
	}

	if lessPointTypes(rune(pointKey1[0]), rune(pointKey2[0])) {
		pointKey1, pointKey2 = pointKey2, pointKey1
	}

	if pointKey1[0] == 'l' { // Country
		point1 := sf.PointMap[pointKey1]
		point2 := sf.PointMap[pointKey2]
		return point1.countryID == point2.countryID
	}

	if pointKey1[0] == 'c' { // City
		// Point2 is a station
		city2 := sf.StationMap[pointKey2]
		return pointKey1 == city2

	}

	if pointKey1[0] == 's' { // Airport
		return true
	}

	// Something unknown
	return false
}

func (sf *SuggestFabric) FindCommonSuggests(tokens []string, q Query) []SuggestMetaInfo {
	oldSuggests := sf.FindSuggestsFromStart(tokens[0], q, -1)
	ind := 1
	for ind < len(tokens) && len(oldSuggests) == 0 {
		oldSuggests = sf.FindSuggestsFromStart(tokens[ind], q, -1)
		ind++
	}

	if len(oldSuggests) == 0 {
		return []SuggestMetaInfo{}
	}

	for _, token := range tokens[ind:] {
		newSuggests := make([]SuggestMetaInfo, 0)
		currentSuggests := sf.FindSuggestsFromStart(token, q, -1)
		for _, suggest := range currentSuggests {
			pointKey := suggest.pointKey
			for _, oldSuggest := range oldSuggests {
				oldPointKey := oldSuggest.pointKey
				if sf.isCommon(oldPointKey, pointKey) {
					suggestToAdd := oldSuggest
					if lessPointTypes(rune(pointKey[0]), rune(oldPointKey[0])) {
						suggestToAdd = suggest
					}
					newSuggests = append(newSuggests, suggestToAdd)
				}
			}

		}
		oldSuggests = newSuggests
	}

	return oldSuggests
}

func (sf *SuggestFabric) TryMixedInput(rawQuery string, q Query) []SuggestMetaInfo {
	tokens := split(rawQuery, []rune(" (),;"))
	if len(tokens) <= 1 {
		return []SuggestMetaInfo{}
	}

	commonSuggests := sf.FindCommonSuggests(tokens, q)
	if len(commonSuggests) != 0 {
		return commonSuggests
	}

	allSuggests := make([]SuggestMetaInfo, 0)
	for _, token := range tokens {
		allSuggests = append(allSuggests, sf.FindSuggestsFromStart(token, q, q.maxCount)...)
	}

	return allSuggests
}

func (sf *SuggestFabric) FindSuggestWithTypo(rawQuery string, q Query) []SuggestMetaInfo {
	var additionalSuggests []SuggestMetaInfo
	ngramIndexes := sf.getNGramIndexesTypos(q)

	runeQuery := []rune(rawQuery)
	queryLen := len(runeQuery)

	for _, ngramIndex := range ngramIndexes {
		lenIndex := ngramIndex.TitlesLenIndex
		leftBorderInIndex := max(0, queryLen-config.Engine.MaxTypoDistance-2)
		if leftBorderInIndex < len(lenIndex) {
			leftIndex := lenIndex[leftBorderInIndex]
			rightIndex := lenIndex[min(len(lenIndex)-1, queryLen+config.Engine.MaxTypoDistance-1)]
			nearest := FindNearest(runeQuery, ngramIndex.Titles[leftIndex:rightIndex], config.Engine.MaxTypoDistance)

			for ind := 0; ind < len(nearest); ind++ {
				if nearest[ind].Distance < config.Engine.MinTypoDistance { // Skip variants without typos
					continue
				}
				currentTitle := string(nearest[ind].Word)
				currentTitle = strings.Replace(currentTitle, "ё", "е", -1)

				currentBranch := ngramIndex.TitlePointsMap[currentTitle]
				additionalSuggests = append(additionalSuggests, currentBranch...)
			}
		}
	}

	for x := 0; x < len(additionalSuggests); x++ {
		additionalSuggests[x].misprint = true
	}

	additionalSuggests = sf.dropWithoutAirports(additionalSuggests)
	return additionalSuggests
}

func (sf *SuggestFabric) FindSuggestWithTypoAll(rawQuery string, q Query) []SuggestMetaInfo {
	var additionalSuggests []SuggestMetaInfo
	ngramIndexes := sf.getNGramIndexesTypos(q)

	runeQuery := []rune(rawQuery)
	queryLen := len(runeQuery)

	for _, ngramIndex := range ngramIndexes {
		lenIndex := ngramIndex.AllInputsLenIndex
		leftBorderInIndex := max(0, queryLen-config.Engine.MaxTypoDistance-2)
		if leftBorderInIndex < len(lenIndex) {
			leftIndex := lenIndex[leftBorderInIndex]
			rightIndex := lenIndex[min(len(lenIndex)-1, queryLen+config.Engine.MaxTypoDistance-1)]
			nearest := FindNearest(runeQuery, ngramIndex.AllInputs[leftIndex:rightIndex], config.Engine.MaxTypoDistance)

			for ind := 0; ind < len(nearest); ind++ {
				if nearest[ind].Distance < config.Engine.MinTypoDistance { // Skip variants without typos
					continue
				}
				currentTitle := string(nearest[ind].Word)
				currentTitle = strings.Replace(currentTitle, "ё", "е", -1)

				currentBranch := ngramIndex.CharsBeginsMap[currentTitle]
				additionalSuggests = append(additionalSuggests, currentBranch...)
			}
		}
	}

	for x := 0; x < len(additionalSuggests); x++ {
		additionalSuggests[x].misprint = true
	}

	additionalSuggests = sf.dropWithoutAirports(additionalSuggests)
	return additionalSuggests
}

func (sf *SuggestFabric) FindDefaultSuggests(q Query) []SuggestMetaInfo {
	// У нас нет TOP15 для станций. Превратим в город.
	otherPoint := q.otherPoint

	if getPointType(otherPoint) == StationType {
		otherPoint = sf.StationMap[otherPoint]
	}

	key := q.nationalVersion + "_" + otherPoint
	suggests := sf.Top15DirectionsMap[key]

	// Добиваем популярными городами нац. версии, если показать совсем нечего
	if len(suggests) == 0 {
		suggestsNational := sf.Top15SettlementsMap[q.nationalVersion]
		suggests = append(suggests, suggestsNational...)
	}

	suggests = sf.dropWithoutAirports(suggests)

	if len(suggests) > q.maxCount {
		suggests = suggests[0:q.maxCount]
	}

	return suggests
}

func (sf *SuggestFabric) getNGramIndexes(q Query) []NGramIndex {
	languages := config.Engine.SuggestLanguages[sf.ReverseLangMap[q.langID]]
	ngramIndexes := make([]NGramIndex, len(languages))
	for i, l := range languages {
		ngramIndexes[i] = sf.NGramIndexes[sf.LangMap[l]]
	}

	return ngramIndexes
}

func (sf *SuggestFabric) getNGramIndexesTypos(q Query) []NGramIndex {
	languages := config.Engine.SuggestLanguagesTypos[sf.ReverseLangMap[q.langID]]
	ngramIndexes := make([]NGramIndex, len(languages))
	for i, l := range languages {
		ngramIndexes[i] = sf.NGramIndexes[sf.LangMap[l]]
	}

	return ngramIndexes
}

func (sf *SuggestFabric) FindBaseSuggests(rawQuery string, q Query, maxSuggest int) []SuggestMetaInfo {
	ngramIndexes := sf.getNGramIndexes(q)

	suggests := make([]SuggestMetaInfo, 0)
	for _, ngramIndex := range ngramIndexes {
		suggests = append(suggests, ngramIndex.CharsBeginsMap[rawQuery]...)
	}

	enoughSuggests := maxSuggest > 0 && len(suggests) >= maxSuggest
	if !enoughSuggests && len([]rune(rawQuery)) >= config.Engine.MinLenForTypo {
		additionalSuggests := sf.FindSuggestWithTypo(rawQuery, q)
		if len(additionalSuggests) > 0 {
			suggests = append(suggests, additionalSuggests...)
		}
	}
	if len(suggests) < maxSuggest || maxSuggest == -1 {
		for _, ngramIndex := range ngramIndexes {
			s := ngramIndex.DifficultCharsMap[rawQuery]
			suggests = append(suggests, s...)
		}
	}

	suggests = sf.dropWithoutAirports(suggests)
	suggests = sf.SortSuggests(suggests, q)
	suggests = sf.UniqSuggests(suggests, maxSuggest)

	return suggests
}

func (sf *SuggestFabric) dropWithoutAirports(suggests []SuggestMetaInfo) []SuggestMetaInfo {
	newSuggests := make([]SuggestMetaInfo, 0)
	for _, suggest := range suggests {
		point := sf.PointMap[suggest.pointKey]
		if point.haveAirport {
			newSuggests = append(newSuggests, suggest)
		}
	}

	return newSuggests
}

// Саджесты, которые начинаются с начала или трудные случаи для всех раскаладок и транслитирации
func (sf *SuggestFabric) FindSuggestsFromStart(rawQuery string, q Query, maxSuggest int) []SuggestMetaInfo {
	suggests := sf.FindBaseSuggests(rawQuery, q, maxSuggest)
	fillPuntedField(suggests, false)

	// Пробуем найти транслитерацию
	if q.lang != "en" {
		trancliteratedRawQuery, ok := Transliterate(rawQuery, "en", q.lang)
		if ok && trancliteratedRawQuery != "" {
			transliteratedSuggests := sf.FindBaseSuggests(trancliteratedRawQuery, q, maxSuggest)
			fillPuntedField(transliteratedSuggests, false)
			suggests = append(suggests, transliteratedSuggests...)
			suggests = sf.UniqSuggests(suggests, maxSuggest)
		}
	}

	// Пробуем найти для неправильной раскладки
	for _, langPair := range config.Engine.PuntoLanguages[q.lang] {
		puntedRawQuery, ok := puntoSwitch(rawQuery, langPair.from, langPair.to)
		if ok && puntedRawQuery != "" {
			puntedSuggests := sf.FindBaseSuggests(puntedRawQuery, q, maxSuggest)
			fillPuntedField(puntedSuggests, true)
			suggests = append(suggests, puntedSuggests...)
		}
	}

	suggests = sf.SortSuggests(suggests, q)
	suggests = sf.UniqSuggests(suggests, maxSuggest)

	return suggests
}

// Посчиаем сколько еще можно саджестов
func getMaxSize(found, maxSuggest int) int {
	if maxSuggest == -1 {
		return -1
	}

	return maxSuggest - found
}

//  Саджесты с середины для всех раскладок и всех языков
func (sf *SuggestFabric) FindSuggestsFromCenter(rawQuery string, q Query, maxSuggest int) []SuggestMetaInfo {
	suggests := sf.FindAdditionalSuggests(rawQuery, q, maxSuggest)
	fillPuntedField(suggests, false)
	// Пробуем найти транслитерацию
	notEnough := len(suggests) < maxSuggest || maxSuggest == -1
	if notEnough && q.lang != "en" {
		trancliteratedRawQuery, ok := Transliterate(q.rawQuery, "en", q.lang)
		if ok && trancliteratedRawQuery != "" {
			maxSize := getMaxSize(len(suggests), maxSuggest)
			transliteratedSuggests := sf.FindAdditionalSuggests(trancliteratedRawQuery, q, maxSize)
			fillPuntedField(transliteratedSuggests, false)
			suggests = append(suggests, transliteratedSuggests...)
			suggests = sf.UniqSuggests(suggests, maxSuggest)
		}
	}

	// Пробуем найти для неправильной раскладки
	notEnough = len(suggests) < maxSuggest || maxSuggest == -1
	if notEnough {
		maxSize := getMaxSize(len(suggests), maxSuggest)
		for _, langPair := range config.Engine.PuntoLanguages[q.lang] {
			puntedRawQuery, ok := puntoSwitch(q.rawQuery, langPair.from, langPair.to)
			if ok && puntedRawQuery != "" {
				puntedSuggests := sf.FindAdditionalSuggests(puntedRawQuery, q, maxSize)
				fillPuntedField(puntedSuggests, true)
				suggests = append(suggests, puntedSuggests...)
			}
		}
		suggests = sf.UniqSuggests(suggests, maxSuggest)
	}
	suggests = sf.SortSuggests(suggests, q)
	suggests = sf.UniqSuggests(suggests, maxSuggest)

	return suggests

}

func (sf *SuggestFabric) FillWithPopularCities(suggests []SuggestMetaInfo, number int) []SuggestMetaInfo {
	newSuggests := make([]SuggestMetaInfo, 0)
	added := 0
	i := 0
	for ; i < len(suggests) && added < number; i++ {
		newSuggests = append(newSuggests, suggests[i])
		if getPointType(suggests[i].pointKey) == CountryType {
			nextPopularity := -1
			if i != len(suggests)-1 {
				nextPopularity = suggests[i+1].popularity
			}

			for _, popularCitySuggest := range sf.PopularCities[suggests[i].pointKey] {
				if popularCitySuggest.popularity >= nextPopularity || nextPopularity == -1 {
					newSuggests = append(newSuggests, popularCitySuggest)
					newSuggests[len(newSuggests)-1].level = 1
					newSuggests[len(newSuggests)-1].added = true
					added += 1
				}

				if added == number {
					break
				}
			}
		}
	}

	if i < len(suggests)-1 {
		newSuggests = append(newSuggests, suggests[i+1:]...)
	}
	return newSuggests
}

func (sf *SuggestFabric) AddAirportsToCities(suggests []SuggestMetaInfo, maxAdd int) []SuggestMetaInfo {
	added := 0
	newSuggests := make([]SuggestMetaInfo, 0)
	i := 0
	for ; i < len(suggests); i++ {
		s := suggests[i]
		newSuggests = append(newSuggests, s)
		if getPointType(s.pointKey) == SettlementType && len(sf.Settlement2AllStations[s.pointKey]) > 1 {
			for _, airportKey := range sf.Settlement2AllStations[s.pointKey] {
				stationSuggest := SuggestMetaInfo{
					pointKey:       airportKey,
					pointTypeOrder: StationType,
					level:          s.level + 1,
					added:          true,
				}
				newSuggests = append(newSuggests, stationSuggest)
				added += 1
				if added == maxAdd {
					break
				}
			}
		}

		if added == maxAdd {
			break
		}
	}

	if i < len(suggests)-1 {
		newSuggests = append(newSuggests, suggests[i+1:]...)
	}

	return newSuggests
}

func (sf *SuggestFabric) FindSuggests(rawQuery string, q Query) []SuggestMetaInfo {
	// Дефолтный саджест
	if rawQuery == "" {
		return sf.FindDefaultSuggests(q)
	}

	// Правильный язык, правильная раскладка
	suggests := sf.FindSuggestsFromStart(rawQuery, q, q.maxCount)

	// Пробуем найти с середины
	if len(suggests) == 0 {
		suggests = sf.FindSuggestsFromCenter(rawQuery, q, q.maxCount)
	}

	// Пробуем разбить вход на несколько объектов
	if len(suggests) == 0 {
		suggests = sf.TryMixedInput(rawQuery, q)
		if len(suggests) > 0 {
			suggests = sf.SortSuggests(suggests, q)
			suggests = sf.UniqSuggests(suggests, q.maxCount)
		}
	}

	if len(suggests) == 0 {
		suggests = sf.FindSuggestWithTypoAll(rawQuery, q)
		suggests = sf.SortSuggests(suggests, q)
		suggests = sf.UniqSuggests(suggests, q.maxCount)
	}

	if q.blended {
		suggests = sf.AddInnerPoints(q, suggests)
	}

	return suggests
}

func (sf *SuggestFabric) AddInnerPoints(q Query, suggests []SuggestMetaInfo) []SuggestMetaInfo {
	if (q.suggestLen >= 2 || q.geobaseID != 0) && q.needStation {
		suggests = sf.AddAirportsToCities(suggests, q.maxCount-len(suggests))
	}
	if len(suggests) < q.maxCount {
		suggests = sf.FillWithPopularCities(suggests, q.maxCount-len(suggests))
	}
	suggests = sf.RemoveDuplicates(suggests)
	if !q.showEqualAirports {
		suggests = sf.RemoveAirportsEqualToCities(suggests, q.lang)
	}
	suggests = sf.SortSuggests(suggests, q)
	suggests = sf.UniqSuggests(suggests, q.maxCount)
	return suggests
}

func (sf *SuggestFabric) FindAdditionalSuggests(rawQuery string, q Query, maxSuggest int) []SuggestMetaInfo {
	ngramIndexes := sf.getNGramIndexes(q)

	suggests := make([]SuggestMetaInfo, 0)
	for _, ngramIndex := range ngramIndexes {
		suggests = append(suggests, ngramIndex.CharsOthersMapAll[rawQuery]...)
	}

	suggests = sf.dropWithoutAirports(suggests)
	suggests = sf.SortSuggests(suggests, q)
	suggests = sf.UniqSuggests(suggests, maxSuggest)

	return suggests
}

func (sf *SuggestFabric) FindSuggestsByGeobaseID(q Query) []SuggestMetaInfo {
	point, ok := sf.GeoID2Point[q.geobaseID]
	if !ok {
		return []SuggestMetaInfo{}
	}

	sm := SuggestMetaInfo{
		pointKey: point,
	}

	return sf.AddInnerPoints(q, []SuggestMetaInfo{sm})
}

// Добавим (проиндексируем) саджест в фабрику
func (sf *SuggestFabric) addSuggest(suggestBody string, pointKey string, lang string, pointTypeOrder int, secondPointTypeOrder int) {
	// Заменим ё на е
	suggestBody = strings.Replace(suggestBody, "ё", "е", -1)

	var titleRunes []rune
	langID := sf.LangMap[lang]

	// Незачем индексировать пустоту
	if suggestBody == "" {
		return
	}

	// Increase ID
	atomic.AddInt64(&sf.SuggestID, 1)

	// Проиндексируем саджест
	titleRunes = []rune(strings.ToLower(suggestBody))

	majority, ok := sf.MajorityMap[pointKey]
	if !ok {
		majority = UnknownSettlementMajority
	}

	sm := SuggestMetaInfo{
		id:                   sf.SuggestID,
		pointKey:             pointKey,
		langID:               langID,
		pointTypeOrder:       pointTypeOrder,
		secondPointTypeOrder: secondPointTypeOrder,
		majority:             majority,
		suggestLen:           len(titleRunes),
		level:                0, // По умолчанию саджесты не вложены
		added:                false,
		haveAirport:          sf.PointMap[pointKey].haveAirport,
		title:                suggestBody,
	}

	sf.IndexRune(titleRunes, sm)

	ngramIndex := sf.NGramIndexes[langID]
	title := string(titleRunes)
	_, ok = ngramIndex.TitlePointsMap[title]
	if ok {
		ngramIndex.TitlePointsMap[title] = append(ngramIndex.TitlePointsMap[title], sm)
	} else {
		ngramIndex.Titles = append(ngramIndex.Titles, titleRunes)
		ngramIndex.TitlePointsMap[title] = []SuggestMetaInfo{sm}
	}

	sf.NGramIndexes[langID] = ngramIndex
}

// Выведем числа в лог
func (sf *SuggestFabric) PrintLoadStatistics() {
	appLogger.Info("Load stat:")
	appLogger.Infof("Point2IATAMap: %v (%v)", len(sf.Point2IATAMap), reflect.TypeOf(sf.Point2IATAMap).Size())
	appLogger.Infof("Point2SirenaMap: %v (%v)", len(sf.Point2SirenaMap), unsafe.Sizeof(sf.Point2SirenaMap))
	appLogger.Infof("PointMap: %v (%v)", len(sf.PointMap), unsafe.Sizeof(sf.PointMap))
	appLogger.Infof("RegionMap: %v (%v)", len(sf.RegionMap), unsafe.Sizeof(sf.RegionMap))
	appLogger.Infof("CountryMap: %v (%v)", len(sf.CountryMap), unsafe.Sizeof(sf.CountryMap))
	appLogger.Infof("PopularDirectionsMap: %v (%v)", len(sf.PopularDirectionsMap), unsafe.Sizeof(sf.PopularDirectionsMap))
	appLogger.Infof("Top15DirectionsMap: %v (%v)", len(sf.Top15DirectionsMap), unsafe.Sizeof(sf.Top15DirectionsMap))
	appLogger.Infof("Top15SettlementsMap: %v (%v)", len(sf.Top15SettlementsMap), unsafe.Sizeof(sf.Top15SettlementsMap))
	appLogger.Infof("Settlement2StationMap: %v (%v)", len(sf.Settlement2StationMap), unsafe.Sizeof(sf.Settlement2StationMap))
}

// Только самое необходимое для сортировки и вывода саджеста в запросе
type SuggestMetaInfo struct {
	id                   int64
	pointKey             string // PointKey; TODO: Подумать, как превратить в int
	langID               int    // Язык
	majority             int    // Значимость
	popularity           int    // Популярность
	directionPopularity  int    // Популярность направления
	pointTypeOrder       int    // Приоритет по типу саджеста: страна, город, аэропорт
	secondPointTypeOrder int    // Вторичная сортировка: IATA страны, города, аэропорта
	sameLang             bool   // Совпадение языка
	isCommonSynonym      bool   // Общий синоним
	likeOther            bool   // Совпадение с "другим" полем
	suggestLen           int    // Длинна исходного саджеста
	likeCode             bool   // Приоритезация кодов
	misprint             bool   // Опечатка
	fullSuggest          bool   // Пользователь ввел весь текст
	level                int64  // Уровень вложенности варианта (начинаем от 0)
	added                bool   // Сами ли мы его добавили
	punted               bool   // Была ли изменена раскладка
	haveAirport          bool   // Если аэропрт у точки
	title                string // Название пункта (отдельно, из-за синонимов)
}

type CachedSuggests struct {
	PlainText string
	Count     int
}

type SuggestMetaInfoSortBlended []SuggestMetaInfo

func (sm SuggestMetaInfoSortBlended) Len() int      { return len(sm) }
func (sm SuggestMetaInfoSortBlended) Swap(i, j int) { sm[i], sm[j] = sm[j], sm[i] }
func (sm SuggestMetaInfoSortBlended) Less(i, j int) bool {
	// Популярность направления
	if sm[i].directionPopularity != sm[j].directionPopularity {
		return sm[i].directionPopularity > sm[j].directionPopularity
	}

	if sm[i].fullSuggest != sm[j].fullSuggest {
		return sm[i].fullSuggest
	}

	// Совпадение по коду (IATA, etc) - всегда в топе
	if (sm[i].langID == 99 || sm[j].langID == 99) && (sm[i].langID != sm[j].langID) {
		// По длинне саджеста
		if sm[i].likeCode != sm[j].likeCode {
			return sm[i].likeCode
		}
	}

	// По языкам (совпадение всегда в топе)
	if sm[i].sameLang != sm[j].sameLang {
		return sm[i].sameLang
	}

	// Пессимизируем совпадения
	if sm[i].likeOther != sm[j].likeOther {
		return !sm[i].likeOther
	}

	// Популярность города
	if sm[i].popularity != sm[j].popularity {
		return sm[i].popularity > sm[j].popularity
	}

	// Общий синоним (совпадение всегда в топе)
	if sm[i].isCommonSynonym != sm[j].isCommonSynonym {
		return sm[i].isCommonSynonym
	}

	if sm[i].haveAirport != sm[j].haveAirport {
		return sm[i].haveAirport
	}

	// Размер города
	return sm[i].majority < sm[j].majority
}

// Сортировка (ранжирование) саджестов
type SuggestMetaInfoSort []SuggestMetaInfo

func (sm SuggestMetaInfoSort) Len() int      { return len(sm) }
func (sm SuggestMetaInfoSort) Swap(i, j int) { sm[i], sm[j] = sm[j], sm[i] }
func (sm SuggestMetaInfoSort) Less(i, j int) bool {
	// Популярность направления
	if sm[i].directionPopularity != sm[j].directionPopularity {
		return sm[i].directionPopularity > sm[j].directionPopularity
	}

	if sm[i].fullSuggest != sm[j].fullSuggest {
		return sm[i].fullSuggest
	}

	// Совпадение по коду (IATA, etc) - всегда в топе
	if (sm[i].langID == 99 || sm[j].langID == 99) && (sm[i].langID != sm[j].langID) {
		// По длинне саджеста
		if sm[i].likeCode != sm[j].likeCode {
			return sm[i].likeCode
		}
	}

	// По языкам (совпадение всегда в топе)
	if sm[i].sameLang != sm[j].sameLang {
		return sm[i].sameLang
	}

	// Пессимизируем совпадения
	if sm[i].likeOther != sm[j].likeOther {
		return !sm[i].likeOther
	}

	// По вторичному типу
	if sm[i].secondPointTypeOrder != sm[j].secondPointTypeOrder {
		return sm[i].secondPointTypeOrder < sm[j].secondPointTypeOrder
	}

	// Популярность города
	if sm[i].popularity != sm[j].popularity {
		return sm[i].popularity > sm[j].popularity
	}

	// Общий синоним (совпадение всегда в топе)
	if sm[i].isCommonSynonym != sm[j].isCommonSynonym {
		return sm[i].isCommonSynonym
	}

	if sm[i].haveAirport != sm[j].haveAirport {
		return sm[i].haveAirport
	}

	// Размер города
	return sm[i].majority < sm[j].majority
}
