# -*- coding: utf-8 -*-
import json
import os
import sys
import time
from collections import Counter
from datetime import datetime

sys.path.append(os.path.dirname(os.path.abspath(__file__)))


TIMELINE_LOG_PATH = '/ephemeral/var/log/yandex-ticket-daemon/yt/timeline.log'
PARTNERS_QUERY_LOG_PATH = '/ephemeral/var/log/yandex-ticket-daemon/yt/partners_query.log'


def to_seconds(date):
    return time.mktime(date.timetuple())


class OffsetLogReader(object):
    def __init__(self, log_path, offset_file_path=None):
        self._log_path = log_path
        self._offset_file_path = offset_file_path or log_path + '_offset'

    def read(self):
        lines = []
        with open(self._log_path, 'r') as log:
            if os.path.isfile(self._offset_file_path):
                log.seek(0, os.SEEK_END)
                end = log.tell()
                log.seek(self._load_offset())
                s = log.tell()
                # usually that means rotated log
                if s > end:
                    log.seek(0)
                lines = log.read().splitlines()
                self._save_offset(log.tell())
            else:
                log.seek(0, os.SEEK_END)
                end = log.tell()
                self._save_offset(end)
        for line in lines:
            yield line

    def _load_offset(self):
        try:
            with open(self._offset_file_path, 'r') as f:
                return int(f.read())
        except ValueError:
            return 0

    def _save_offset(self, offset):
        with open(self._offset_file_path, 'w') as f:
            f.write(str(offset))


class BaseLogParser(object):
    def __init__(self, logger, registry):
        self._logger = logger
        self._registry = registry
        self._requests_counter = Counter()
        self._timings_histograms = {}

    def _get_timings_histogram(self, labels):
        k = tuple(sorted(labels.items()))
        if k not in self._timings_histograms:
            self._timings_histograms[k] = self._registry.histogram_rate(
                labels,
                'exponential',
                base=1.3,
                bucket_count=48,
                scale=1,
            )

        return self._timings_histograms[k]


class TimelineLogParser(BaseLogParser):
    def _parse_line(self, line):
        """
            Формат:
            {"ts": "2020-01-13 09:56:48", "tid": "13:065648.000000", "elapsed_ms": 606, "event": "start_import", "qid": "200113-095648-173.m_avia.plane.c10310_s9600213_2020-01-27_None_economy_1_0_0_ru.ru", "service": "m_avia", "importer": "azimuth"}
            События:
                - start_import
                - first_partner
                - last_partner
        :param line:
        :return:
        """
        return json.loads(line)

    def pull(self, ts, consumer):
        log_reader = OffsetLogReader(TIMELINE_LOG_PATH)
        for line in log_reader.read():
            record = self._parse_line(line)
            if record['event'] == 'spawned_importers':
                continue

            ts_ = float(to_seconds(
                datetime.strptime(record['ts'], '%Y-%m-%d %H:%M:%S')
            ))
            event = str(record['event'])
            importer = str(record['importer'])
            elapsed_ms = record['elapsed_ms']

            labels = {
                'sensor': 'timings',
                'event': event,
                'importer': importer,
            }
            consumer.gauge(labels, ts_, elapsed_ms)
            labels['sensor'] = 'timings-histogram'
            self._get_timings_histogram(labels).collect(int(elapsed_ms))

            self._requests_counter[(event, importer)] += 1

        for (event, importer), value in self._requests_counter.iteritems():
            consumer.rate({
                'sensor': 'count',
                'event': event,
                'importer': importer,
            }, 0, value)


class PartnersQueryLogParser(BaseLogParser):
    def _parse_line(self, line):
        """
            Формат:
            tskv    tskv_format=rasp-partners-query-log     status=got_reply        query_time=5145.35713196        fetch_time=6013.52596283        qid=200119-012929-199.yeah_42_tails.plane.c239_c172_2020-02-14_2020-02-21_economy_1_0_0_ru.ru   variants_len=521        importer=biletik5[biletikaeroag]        unixtime=1579386603     init_id=200119-012929-198.yeah_42_tails.plane.c239_c172_2020-02-14_2020-02-21_economy_1_0_0_ru.ru       packing_time=27644.0901756
            got_reply
            timeout - опрашивали партнера дольше 180 секунд
            response_error - получили в ответе ошибку в ожидаемом виде,
            got_failure - получили неожиданную ошибку при парсинге вариантов или при запросе к партнеру
            query_invalid - не отправляем запрос в партнерам по ограничениям модуля импорта
        :param line:
        :return:
        """

        return dict(item.split('=', 1) for item in line.split('\t')[1:])

    def pull(self, ts, consumer):
        log_reader = OffsetLogReader(PARTNERS_QUERY_LOG_PATH)
        for line in log_reader.read():
            record = self._parse_line(line)

            ts_ = float(record['unixtime'])
            status = record['status']
            partner_code = self.parse_partner_code(record['importer'])

            query_time = float(record['query_time'])

            labels = {
                'sensor': 'timings',
                'status': status,
                'partner_code': partner_code,
            }
            consumer.gauge(labels, ts_, query_time)
            labels['sensor'] = 'timings-histogram'
            self._get_timings_histogram(labels).collect(int(query_time))

            self._requests_counter[(status, partner_code)] += 1

        for (status, partner_code), value in self._requests_counter.iteritems():
            consumer.rate({
                'sensor': 'count',
                'status': status,
                'partner_code': partner_code,
            }, 0, value)

    @staticmethod
    def parse_partner_code(importer):
        if importer.startswith('amadeus'):
            return 'amadeus'
        elif importer.startswith('dohop'):
            return 'dohop'
        else:
            return importer[importer.find('[') + 1: -1]
