# -*- coding: utf-8 -*-
import gc
import os
import socket

import django

from travel.library.python.gunicorn.worker_watcher import WorkersWatcher
from travel.library.python.solomon.server import SolomonMetricsServer

from travel.avia.library.python.redis.ping import RedisBackgroundPing
from travel.avia.library.python.shared_objects.shared_flag import SharedFlag

from travel.avia.ticket_daemon.ticket_daemon.api.cache import default_cache, shared_cache
from travel.avia.ticket_daemon.ticket_daemon.settings.application import PARTNER_QUERY_TIMEOUT, get_metric_queue
from travel.avia.ticket_daemon.ticket_daemon.settings.caches import (
    CACHEROOT,
    CACHE_REDIS,
    MULTI_CACHE,
    REDIS_MAXIMUM_LAG,
    REDIS_PING_SLEEP_TIME,
    SHARED_CACHE_REDIS,
)
from travel.avia.ticket_daemon.ticket_daemon.settings.environment import env_variable_provider


name = 'yandex-ticket-daemon'

# Тут нужно аккуратно, права на запись затираются после рестарта машины
# >> because /run/ in mounted as tmpfs and cleared after each server reboot.
# В нашем случае каталог с правами создаётся из скрипта запуска самого тд
bind = env_variable_provider.get('BIND_ADDRESS')

loglevel = env_variable_provider.get('LOG_LEVEL')
errorlog = env_variable_provider.get('GUNICORN_LOG')
capture_output = (env_variable_provider.get('GUNICORN_CAPTURE_OUTPUT', required=False) or 'true') == 'true'

workers = int(env_variable_provider.get('GUNICORN_WORKERS_COUNT'))
worker_connections = 1000
threads = int(env_variable_provider.get('GUNICORN_THREADS_COUNT', required=False, default='1'))
max_requests = 100000
max_requests_jitter = max_requests // 5
worker_class = env_variable_provider.get(
    'GUNICORN_WORKER_CLASS',
    required=False,
    default='travel.library.python.gunicorn.worker_watcher.workers.SyncWorker',
)
timeout = int(env_variable_provider.get('GUNICORN_WORKER_TIMEOUT', default=PARTNER_QUERY_TIMEOUT+20, required=False))
preload_app = True

VERSION = os.getenv('DEPLOY_DOCKER_IMAGE', ':unknown').rsplit(':', 1)[-1]
TRACING_SERVICE_NAME = 'avia-ticket-daemon'
TRACING_JAEGER_SAMPLER_TYPE = os.getenv('JAEGER_SAMPLER_TYPE', 'probabilistic')
TRACING_JAEGER_SAMPLER_PARAMETER = float(os.getenv('JAEGER_SAMPLER_PARAMETER', 0.001))
DISABLE_TRACING = bool(int(os.getenv('DISABLE_TRACING', False)))

shutdown_flag = SharedFlag()
cache_readiness_flag = SharedFlag()
shared_cache_readiness_flag = SharedFlag()

solomon_server = SolomonMetricsServer()
if (CACHE_REDIS or SHARED_CACHE_REDIS) and not MULTI_CACHE:
    solomon_server.register_metric_queue(get_metric_queue())

key_ttl = 60
if CACHE_REDIS and not MULTI_CACHE:
    cache_ping = RedisBackgroundPing(
        default_cache,
        '{}ping_check/{}'.format(CACHEROOT, socket.gethostname()),
        key_ttl,
        REDIS_PING_SLEEP_TIME,
        REDIS_MAXIMUM_LAG,
        cache_readiness_flag,
    )
if SHARED_CACHE_REDIS and not MULTI_CACHE:
    shared_cache_ping = RedisBackgroundPing(
        shared_cache,
        '{}shared_ping_check/{}'.format(CACHEROOT, socket.gethostname()),
        key_ttl,
        REDIS_PING_SLEEP_TIME,
        0,
        shared_cache_readiness_flag,
    )

watcher = WorkersWatcher()


def _precache(server):
    from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import closing_db
    from travel.avia.library.python.common.precache.backend import precache
    from travel.avia.ticket_daemon.ticket_daemon.api.models_utils import warm_up_cache
    from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.station import monkey_patch_station_iata
    from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
    from travel.avia.library.python.ticket_daemon.memo import reset_all_caches

    reset_all_caches()
    with closing_db(force=True, logger=server.log):
        server.log.info('Call old precache...')
        precache()
        server.log.info('...Finish old precache')

        server.log.info('Warming up...')
        warm_up_cache(logger=server.log)
        monkey_patch_station_iata()
        partner_secret_storage.recache()
        server.log.info('...Finish Warming up')

    server.log.info('Precache done')


def _init_logbroker_writers(server, create_new=False):
    from travel.avia.ticket_daemon.ticket_daemon.daemon.logbroker import get_processed_qid_writer

    server.log.info('Init logbroker writers...')
    try:
        get_processed_qid_writer(create_new).__enter__()
    except Exception:
        server.log.exception('Failed to init logbroker writers')
        raise
    server.log.info('Done logbroker writers init successfully')


def _heat_partners(server):
    import itertools
    import datetime
    from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.partners import (
        get_actual_partners_without_cache,
        partners_cache,
        _gap_cache_key,
    )
    from travel.avia.ticket_daemon.ticket_daemon.settings import AVIA_NATIONAL_VERSIONS

    server.log.info('Heating partners cache')
    for nv, from_rasp, is_mobile in itertools.product(AVIA_NATIONAL_VERSIONS, [True, False], [True, False]):
        key = _gap_cache_key(national_version=nv, t_code='plane', from_rasp=from_rasp, mobile=is_mobile)
        old_ttl = partners_cache.get_key_ttl(key)
        if old_ttl is None or old_ttl < datetime.timedelta(0):
            server.log.warning("Key %s is expired, overdue %s", key, old_ttl)
        partners_cache[key] = get_actual_partners_without_cache(
            national_version=nv, t_code='plane', from_rasp=from_rasp, mobile=is_mobile
        )
    server.log.info('Heating completed')


def _heat_cache(server):
    import gevent
    import schedule

    from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.partners import TTL

    schedule.every(max(TTL // 2, 1)).seconds.do(_heat_partners, server)
    while True:
        schedule.run_pending()
        gevent.sleep(1)


def on_starting(server):
    django.setup()

    server.log.info('start master process...')

    watcher.on_starting(server)

    solomon_server.register_gauge({'sensor': 'gunicorn.requests_in_work'}, watcher.get_requests_in_work_count)
    solomon_server.register_gauge({'sensor': 'gunicorn.requests_in_work_max'}, watcher.get_max_requests_in_work)
    solomon_server.run_in_thread()

    if CACHE_REDIS and not MULTI_CACHE:
        cache_ping.run()
    if SHARED_CACHE_REDIS and not MULTI_CACHE:
        shared_cache_ping.run()
    _precache(server)
    server.log.info('... finish start master process')


def on_reload(server):
    server.log.info('Update cache ...')
    if CACHE_REDIS and not MULTI_CACHE:
        cache_ping.run()
    if SHARED_CACHE_REDIS and not MULTI_CACHE:
        shared_cache_ping.run()
    _precache(server)
    server.log.info('... finish update cache')


def on_exit(server):
    if CACHE_REDIS and not MULTI_CACHE:
        cache_ping.stop()
    if SHARED_CACHE_REDIS and not MULTI_CACHE:
        shared_cache_ping.stop()
    solomon_server.stop_thread()


def pre_fork(server, worker):
    """
    Закрываем соединение, чтобы не передавать воркерам
    копию сокета [RASPWIZARDS-175]
    """
    from django.core.cache import cache
    cache.close()

    watcher.pre_fork(server, worker)

    # собираем мусор, чтобы не оставлять его в форках
    gc.collect()


def post_fork(server, worker):
    import threading
    from flask_opentracing_helpers import setup_tracing
    from opentracing.scope_managers.gevent import GeventScopeManager

    from travel.avia.ticket_daemon.ticket_daemon.lib import feature_flags
    from travel.avia.ticket_daemon.wsgi import application

    application.shutdown_flag = shutdown_flag
    application.cache_readiness_flag = cache_readiness_flag
    application.shared_cache_readiness_flag = shared_cache_readiness_flag
    if not CACHE_REDIS or MULTI_CACHE:
        application.cache_readiness_flag.set()
    if not SHARED_CACHE_REDIS or MULTI_CACHE:
        application.shared_cache_readiness_flag.set()

    _init_logbroker_writers(server)

    server.log.info('Running heater')
    if feature_flags.heat_partners_cache():
        heater = threading.Thread(target=_heat_cache, args=(server,))
        heater.setDaemon(True)
        heater.start()

    if DISABLE_TRACING:
        server.log.info('Tracing is disabled')
        return

    server.log.info('Setup tracing...')
    env_variables = ['DEPLOY_STAGE_ID', 'DEPLOY_NODE_DC', 'DEPLOY_BOX_ID']
    try:
        setup_tracing(
            flask_application=application,
            service_name=TRACING_SERVICE_NAME,
            version=VERSION,
            include_env_variables=env_variables,
            filter_paths=['service routes.ping'],
            scope_manager=GeventScopeManager(),
            instrument_libraries_with_tracing=False,
            jaeger_sampler_type=TRACING_JAEGER_SAMPLER_TYPE,
            jaeger_sampler_parameter=TRACING_JAEGER_SAMPLER_PARAMETER,
            jaeger_client_enable_logging=False,
        )
        _install_compatible_patches()
    except Exception:
        server.log.exception('Failed to setup tracing')
    else:
        server.log.info('Done setup tracing successfully')


def _install_compatible_patches():
    """
    Копипаста из opentracing_instrumentation.client_hooks.install_all_patches
    но только совместимые с нашим приложением патчи
    """
    from opentracing_instrumentation.client_hooks import (
        mysqldb,
        psycopg2,
        strict_redis,
        sqlalchemy,
        tornado_http,
        urllib,
        urllib2,
        requests,
    )

    mysqldb.install_patches()
    psycopg2.install_patches()
    strict_redis.install_patches()
    sqlalchemy.install_patches()
    tornado_http.install_patches()
    urllib.install_patches()
    urllib2.install_patches()
    requests.install_patches()
