# -*- coding: utf-8 -*-
import logging
import ujson as json
import zlib

from django.core.cache import caches

from travel.avia.library.python.common.utils.memcache_backend import MemcachedCache
from travel.avia.library.python.ticket_daemon.decorators import log_elapsed_time
from travel.avia.ticket_daemon.ticket_daemon.settings.caches import MULTI_CACHE

logger = logging.getLogger(__name__)


def decorate_methods(methods, decorator):
    def decorate(cls):
        for method in methods:
            setattr(cls, method, decorator(getattr(cls, method)))
        return cls
    return decorate


@decorate_methods(('set', 'get', 'set_many', 'get_many'), log_elapsed_time(logger))
class LoggingMemcachedCache(MemcachedCache):
    """
    Бэкенд для memcache, логирующий времена set, get
    """

    def unpack(self, packed):
        return json.loads(zlib.decompress(packed))


class MultiCache(object):
    def __init__(self, daemon_memcache, daemon_redis_cache):
        self.memcache = daemon_memcache
        self.redis = daemon_redis_cache

    @log_elapsed_time(logger, statsd_prefix='redis.set.elapsed')
    def set(self, key, value, timeout=None):
        self.redis.set(key, value, timeout)
        self.memcache.set(key, value, timeout)

    @log_elapsed_time(logger, statsd_prefix='redis.get.elapsed')
    def get(self, key):
        return self.memcache.get(key)

    @log_elapsed_time(logger, statsd_prefix='redis.set_many.elapsed')
    def set_many(self, data, timeout=None):
        self.redis.set_many(data, timeout)
        self.memcache.set_many(data, timeout)

    @log_elapsed_time(logger, statsd_prefix='redis.get_many.elapsed')
    def get_many(self, keys):
        return self.memcache.get_many(keys)

    def unpack(self, packed):
        return json.loads(zlib.decompress(packed))


default_cache = caches['default']
if MULTI_CACHE:
    shared_cache = MultiCache(caches['memcached'], caches['shared_cache'])
else:
    shared_cache = caches['shared_cache']
