# -*- coding: utf-8 -*-
from __future__ import division

import hashlib
import logging
from datetime import timedelta
from itertools import chain
from typing import Optional, List

import pytz
from django.utils.functional import cached_property
from django.utils.encoding import force_unicode, force_str

from travel.avia.library.python.common.lib.company_finder import CompanyFinder
from travel.avia.library.python.common.models.partner import Partner
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.utils.caching import cache_method_result
from travel.avia.library.python.common.utils.date import MSK_TZ, UTC_TZ
from travel.avia.library.python.common.utils.iterrecipes import pairwise
from travel.avia.library.python.price_prediction import PriceCategory

from travel.avia.library.python.ticket_daemon.date import unixtime
from travel.avia.library.python.ticket_daemon.memo import memoize
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.region import get_region_by_id
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.company import (
    get_aviacompany_by_company_id,
    get_tariffs_by_aviacompany_pk,
    make_companies_by_iata,
    make_companies_by_sirena,
    make_corrections_by_iata,
    make_company_by_id,
)
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.currency import get_currency_by_id
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.settlement import get_settlement_by_id
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.station import (
    get_station_by_iata_or_sirena,
    get_airport_by_id,
)
from travel.avia.ticket_daemon.ticket_daemon.lib import feature_flags
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage_corrector import baggage_corrector
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage_merger import baggage_merger

log = logging.getLogger(__name__)
flight_board_log = logging.getLogger(__name__ + '.flight_board')
flight_board_collection_log = logging.getLogger('flight_board_collection')

SERVICE_CLASS_PRIORITIES = {'economy': 1, 'business': 2, 'first': 3}
MIN_STAY_TIME = timedelta(hours=1)
PARTNERS_BOOKING_AVAILABLE_HOURS = {'pilotua': 4.5}


def min_class(klasses, default='economy'):
    if not klasses:
        return default

    return min(klasses, key=lambda c: SERVICE_CLASS_PRIORITIES[c])


class OperatingFlight(object):
    def __init__(self, company_iata=None, number=None, pure_number=None):
        # type: (str, Optional[str], Optional[str])->None
        """
        :param company_iata: like: "SU" or "ПБД"
        :param number: like "SU 1234Q"
        :param pure_number: number without company iata (like "1234Q")
        """
        self.company_iata = force_unicode(company_iata)
        self.number = (  # format: "SU 1234Q"
            force_unicode(number)
            if number
            else u'{} {}'.format(
                force_unicode(company_iata),
                force_unicode(pure_number),
            )
        )

        self._company = None  # filled on access via company property

    @cached_property
    def company(self):
        # type: ()->Company
        self.complete_company()
        return self._company

    def __eq__(self, other):
        return self.company_iata == other.company_iata and self.number == other.number

    def __ne__(self, other):
        return not self.__eq__(other)

    def complete_company(self):
        self._company = CompanyFinder(
            companies_by_iata=make_companies_by_iata(),
            companies_by_sirena=make_companies_by_sirena(),
            companies_by_icao={},  # make_companies_by_icao(),
            companies_by_icao_ru={},  # make_companies_by_icao_ru(),
            corrections_by_iata=make_corrections_by_iata(),
            company_by_id=make_company_by_id(),
        ).find(self.company_iata, self.number)

        if not self._company:
            log.warn('can not complete operating company for [%r] [%r]', self.company_iata, self.number)
            return

        # Заменим код компании в номере рейса
        if self.number:
            code = self._company.iata or self._company.sirena_id or self._company.icao or ''
            if code:
                number_parts = self.number.split()
                number_parts[0] = code
                self.number = ' '.join(number_parts)
            else:
                log.critical('Company without any code: %s', self._company.id)

    def __repr__(self):
        return '%s(company_iata=\'%s\', number=\'%s\')' % (
            self.__class__.__name__,
            self.company_iata,
            self.number,
        )

    def check_correctness(self, log_level):
        if not self.company:
            log.log(
                log_level,
                'Operating flight object %s is filled incorrectly: company is empty',
                self,
                exc_info=True,
            )
            return False
        if not self.number:
            log.log(
                log_level,
                'Operating flight object %s is filled incorrectly: flight number is empty',
                self,
                exc_info=True,
            )
            return False

        return True


def try_create_operating_flight(company_iata=None, number=None, pure_number=None):
    op_flight = OperatingFlight(company_iata=company_iata, number=number, pure_number=pure_number)
    op_flight.complete_company()
    if not op_flight.check_correctness(logging.WARNING):
        return None

    return op_flight


class Flight(object):
    baggage = None
    company = None
    number = None
    avia_company = None
    fare_code = None
    company_tariff = None

    t_model = None
    charter = None

    _completed = False

    def __init__(self, **kwargs):
        self.station_from = None
        self.station_to = None
        self.departure = None
        self.arrival = None
        self.company = None
        self.operating = None  # type: Optional[OperatingFlight]
        self.klass = None

    @property
    def title(self):
        points = self.points_for_title()
        return "%s - %s" % (
            points['from'].L_title(),
            points['to'].L_title(),
        )

    def points_for_title(self):
        station_from = self.station_from
        station_to = self.station_to

        return {
            'from': station_from.settlement or station_from,
            'to': station_to.settlement or station_to,
        }

    @property
    def msk_departure(self):
        return self.departure and self.departure.astimezone(MSK_TZ).replace(tzinfo=None)

    @property
    def msk_arrival(self):
        return self.arrival and self.arrival.astimezone(MSK_TZ).replace(tzinfo=None)

    @property
    def utc_departure(self):
        return self.departure and self.departure.astimezone(UTC_TZ).replace(tzinfo=None)

    @property
    def utc_arrival(self):
        return self.arrival and self.arrival.astimezone(UTC_TZ).replace(tzinfo=None)

    @property
    def duration(self):
        return self.arrival and self.departure and self.arrival - self.departure

    @cache_method_result
    def cache_key(self):
        # Обязательные параметры без которых кэширование не имеет смысла
        if not self.company or not self.number:
            return None

        return "%s_%s_%s_%s_%s_%s" % (
            self.number,
            self.company.id if self.company else 'None',
            self.local_departure.strftime('%Y-%m-%d %H:%M:%S'),
            self.local_arrival.strftime('%Y-%m-%d %H:%M:%S'),
            self.station_from.id if self.station_from else 'None',
            self.station_to.id if self.station_to else 'None',
        )

    @cached_property
    def key(self):
        return '-'.join(
            map(
                str,
                (
                    _to_str(self.number),
                    self.company and self.company.id,
                    self.station_from and self.station_from.id,
                    self.station_to and self.station_to.id,
                    self.local_departure and self.local_departure.strftime('%m%d%H%M'),
                    self.local_arrival and self.local_arrival.strftime('%m%d%H%M'),
                ),
            )
        )

    def __repr__(self):
        return "<%s %r-%r (%r-%r) [%s]>" % (
            self.__class__.__name__,
            self.station_from,
            self.station_to,
            self.departure,
            self.arrival,
            getattr(self, 'partner', None),
        )


class IATAFlight(Flight):
    local_departure = None
    local_arrival = None
    fare_code = None
    company_iata = None
    station_from_iata = None
    station_to_iata = None
    t_model_name = None

    station_from = None
    station_to = None

    @cached_property
    def key(self):
        return self.make_flight_key(self.local_departure, self.local_arrival, self.number)

    @staticmethod
    def make_flight_key(local_departure, local_arrival, number):
        return '{}{}{}'.format(
            local_departure.strftime('%y%m%d%H%M') if local_departure else '',
            force_str(number.replace(' ', '')),
            local_arrival.strftime('%y%m%d%H%M') if local_arrival else '',
        )

    @cached_property
    def flight_tag(self):
        return self.make_flight_tag(self.local_departure, self.number)

    @staticmethod
    def make_flight_tag(local_departure, number):
        return '{}{}'.format(
            local_departure.strftime('%y%m%d%H%M') if local_departure else '', force_str(number.replace(' ', ''))
        )

    def set_number(self, number):
        self.number = number
        if 'flight_tag' in self.__dict__:
            del self.__dict__['flight_tag']

    def preprocess(self):
        """
        Вызывается из daemon/importer.py:Importer._got_chunk
        Может быть, не нужен?
        """
        try:
            company_code, num = self.number.split(None, 1)
        except Exception as exc:
            log.exception('Bad number[%r]: %s', getattr(self, 'number', 'no number'), exc)
        else:
            new_number = '%s %s' % (company_code, num)
            if self.number != new_number:
                log.info('Changed number from %r to %r', self.number, new_number)
                self.number = new_number

    @property
    def is_complete(self):
        return self.station_from and self.station_to

    def _complete(self):
        self.complete_company()
        if self.operating:
            self.operating.complete_company()

        self.station_from = get_station_by_iata_or_sirena(self.station_from_iata)
        self.station_to = get_station_by_iata_or_sirena(self.station_to_iata)

        if self.station_from and self.local_departure:
            self.departure = _localize_by_station(self.local_departure, self.station_from)

        if self.station_to and self.local_arrival:
            self.arrival = _localize_by_station(self.local_arrival, self.station_to)

    def complete(self):
        if self._completed:
            return
        self._complete()
        self._completed = True

    @property
    def completed(self):
        return self._completed

    def complete_company(self):
        self.company = CompanyFinder(
            companies_by_iata=make_companies_by_iata(),
            companies_by_sirena=make_companies_by_sirena(),
            companies_by_icao={},  # make_companies_by_icao(),
            companies_by_icao_ru={},  # make_companies_by_icao_ru(),
            corrections_by_iata=make_corrections_by_iata(),
            company_by_id=make_company_by_id(),
        ).find(self.company_iata, self.number)
        self.avia_company = None
        self.company_tariff = None

        if not self.company:
            log.warn('can not complete company for [%r] [%r]', self.company_iata, self.number)
            return

        # Возможно с компанией связана авиакомпания
        self.avia_company = get_aviacompany_by_company_id(self.company.id)
        if self.avia_company:
            # Заполним тариф по коду тарифа авиакомпании
            self.company_tariff = get_aviacompany_tariff_by_fare_code(self.avia_company, self.fare_code)
            if not self.company_tariff:
                log.warning('No tariff for %d', self.avia_company.rasp_company_id)

        # Заменим код компании в номере рейса
        c = self.company
        if self.number:
            code = c.iata or c.sirena_id or c.icao or ''
            if code:
                number_parts = self.number.split()
                number_parts[0] = code
                self.number = ' '.join(number_parts)
            else:
                log.critical('Company without any code: %s', c.id)

    @cached_property
    def t_type(self):
        return _t_type('plane')

    def get_baggage_by_company_tariff(self):
        return baggage_corrector.correct_baggage(self)


class Segment(object):
    def __init__(
        self, flight,
        baggage=None, t_model_name=None, klass=None, selfconnect=None, fare_family=None, charter=None
    ):
        assert isinstance(flight, IATAFlight)
        self._flight = flight
        self.baggage = baggage if baggage is not None else Baggage.from_partner()
        self.t_model_name = t_model_name
        self.klass = klass
        self.selfconnect = selfconnect
        self.fare_family = fare_family
        self.charter = charter

    def __repr__(self):
        return '{}:{}'.format(self.__class__, self.__dict__)

    def complete(self, partner_code, is_charter):
        self._flight.complete()
        self.complete_baggage(partner_code, is_charter)

    def complete_baggage(self, partner_code, is_charter):
        self.baggage = baggage_merger.merge(
            partner_code=partner_code,
            flight_number=self._flight.number,
            airline=self._flight.avia_company,
            default_baggage=self._flight.get_baggage_by_company_tariff(),
            partner_baggage=self.baggage,
            company=self._flight.company,
            is_charter=is_charter,
        )

    @property
    def departure(self):
        return self._flight.departure

    @property
    def arrival(self):
        return self._flight.arrival

    @property
    def local_departure(self):
        return self._flight.local_departure

    @property
    def local_arrival(self):
        return self._flight.local_arrival

    @property
    def number(self):
        return self._flight.number

    @property
    def company_iata(self):
        return self._flight.company_iata

    @property
    def operating(self):
        return self._flight.operating

    @property
    def station_from_iata(self):
        return self._flight.station_from_iata

    @property
    def station_to_iata(self):
        return self._flight.station_to_iata

    @property
    def station_from(self):
        return self._flight.station_from

    @property
    def station_to(self):
        return self._flight.station_to

    @property
    def fare_code(self):
        return self._flight.fare_code

    @property
    def flight_tag(self):
        return self._flight.flight_tag

    @property
    def key(self):
        return self._flight.key

    @property
    def company(self):
        return self._flight.company

    @property
    def avia_company(self):
        return self._flight.avia_company

    @property
    def company_tariff(self):
        return self._flight.company_tariff

    @property
    def t_type(self):
        return self._flight.t_type

    @property
    def is_complete(self):
        return self._flight.is_complete

    @property
    def completed(self):
        return self._flight.completed

    def preprocess(self):
        self._flight.preprocess()

    def set_number(self, number):
        return self._flight.set_number(number)


# ======== Заполнение времён ========


@memoize(lambda local_dt, station: (local_dt, station.id))
def _localize_by_station(local_dt, station):
    # return station.localize(loc=local_dt)

    # if local_dt.tzinfo is None:
    #     return station.pytz.localize(local_dt)
    # return local_dt.astimezone(station.pytz)

    # def get_station_tzname(station)
    #     if hasattr(station, 'settlement') and station.settlement:
    #         time_zone = station.settlement.time_zone
    #     if time_zone is None:
    #         time_zone = station.time_zone
    #     if time_zone is None and hasattr(station, 'region'):
    #         time_zone = station.region and station.region.time_zone

    # station_pytz = _get_pytz(get_station_tzname(station))

    station_pytz = _get_pytz(_station_tz_name(station))
    if not station_pytz:
        return None

    if local_dt.tzinfo is None:
        return station_pytz.localize(local_dt)
    return local_dt.astimezone(station_pytz)


@memoize(lambda tzname: tzname)
def _get_pytz(tzname):
    if not tzname:
        return None
    try:
        return pytz.timezone(tzname)
    except pytz.UnknownTimeZoneError:
        log.exception('Unknown pytz TimeZone %s', tzname)
        return None


@memoize(lambda station: station.id)
def _station_tz_name(station):
    # if hasattr(station, 'settlement') and station.settlement:
    if station.settlement_id:
        settlement = get_settlement_by_id(station.settlement_id)
        if settlement:
            time_zone = settlement.time_zone
            if time_zone:
                return time_zone

    time_zone = station.time_zone
    if time_zone:
        return time_zone

    if station.region_id:
        region = get_region_by_id(station.region_id)
        if region:
            return region.time_zone


# ===================================


@memoize(lambda code: code)
def _t_type(code):
    return TransportType.objects.get(code=code)


class FlightFabric(object):
    def __init__(self):
        self._cache = {}  # type: Dict[tuple, IATAFlight]

    def create(
        self,
        company_iata,
        station_from_iata,
        station_to_iata,
        local_departure,
        local_arrival,
        number=None,
        pure_number=None,
        operating=None,  # type: OperatingFlight
        fare_code=None,
        fare_family=None,
        t_model_name=None,
        klass=None,
        charter=None,
        baggage=None,
        selfconnect=None,
        **kwargs
    ):
        assert number or pure_number, 'number or pure_number is not specified.'
        number = (
            force_unicode(number)
            if number
            else u'{} {}'.format(
                force_unicode(company_iata),
                force_unicode(pure_number),
            )
        )

        key = (
            company_iata,
            number,
            station_from_iata,
            station_to_iata,
            local_departure,
            local_arrival,
            fare_code,
        )

        if key not in self._cache:
            f = IATAFlight()

            f.company_iata = company_iata
            f.number = number
            f.station_from_iata = station_from_iata
            f.station_to_iata = station_to_iata
            f.local_departure = local_departure
            f.local_arrival = local_arrival
            f.fare_code = fare_code
            f.operating = operating
            for attr, value in kwargs.iteritems():
                setattr(f, attr, value)
            self._cache[key] = f

        # Operating flight might not be filled in every flight. Update it in cache anyway
        if not self._cache[key].operating and operating:
            self._cache[key].operating = operating

        return Segment(
            self._cache[key],
            baggage=baggage,
            t_model_name=t_model_name,
            klass=klass,
            selfconnect=selfconnect,
            fare_family=fare_family,
            charter=charter,
        )

    def get_flights(self):
        return self._cache.values()


class Flights(object):
    __slots__ = ['segments']

    def __init__(self):
        self.segments = []  # type: List[Segment]

    def __nonzero__(self):
        return bool(self.segments)

    @cached_property
    def key(self):
        return _flights_path_key(self.segments)

    def check_transfers_times(self):
        return all(
            arr.local_arrival and dep.local_departure and arr.local_arrival <= dep.local_departure
            for arr, dep in pairwise(self.segments)
        )

    @property
    def local_departure(self):
        return self.segments[0].local_departure if self.segments else None

    @property
    def local_arrival(self):
        return self.segments[-1].local_arrival if self.segments else None

    @property
    def departure(self):
        return self.segments[0].departure if self.segments else None

    @property
    def arrival(self):
        return self.segments[-1].arrival if self.segments else None


def _flights_path_key(flights):
    return '-'.join(_flight_key(f) for f in flights)


def _flight_key(flight):
    parts = _flight_key_tuple(flight)
    unicode_parts = [_to_unicode(p) for p in parts]
    return u'.'.join(unicode_parts).encode('utf-8')


def _to_str(txt):
    try:
        return '{}'.format(txt)
    except UnicodeEncodeError:
        return unicode(txt).encode('utf-8')


def _to_unicode(txt):
    try:
        return u'{}'.format(txt)
    except UnicodeDecodeError:
        return txt.decode('utf-8')


def _flight_key_tuple(flight):
    return (
        flight.number,
        flight.local_departure.strftime('%m%dT%H%M') if flight.local_departure else '',
    )


def _get_country_id_by_settlement_id(settlement_id):
    if not settlement_id:
        return None
    settlement = get_settlement_by_id(settlement_id)
    if settlement:
        return settlement.country_id


class Variant(object):
    raw_data = ''
    tag = None
    national_tariff = None
    charter = None
    tariff = None
    klass = None
    partner = None
    price_category = PriceCategory.UNKNOWN

    def __init__(self, **kwargs):
        self.forward = Flights()
        self.backward = Flights()
        self.klass = None
        self.order_data = {}
        self.created_at = unixtime()
        self.raw_tariffs = dict()

    @cached_property
    def tag(self):
        forward_tags = (s.flight_tag for s in self.forward.segments)
        backward_tags = (s.flight_tag for s in self.backward.segments)
        return self._make_tag(
            forward_tags,
            backward_tags,
            self.klass,
            self.partner.code,
            self.by_fare_code,
            self.with_baggage,
        )

    @staticmethod
    def _make_tag(forward_tags, backward_tags, klass, partner_code, fare_codes, with_baggage):
        if feature_flags.store_min_tariff_per_fare_code():
            tag_data = [
                '|'.join(forward_tags),
                '|'.join(backward_tags),
                klass,
                partner_code,
                '|'.join(fare_codes),
                with_baggage,
            ]
            tag_text = '{}@{}@{}@{}@{}@{}'.format(*tag_data)
            return hashlib.md5(tag_text).hexdigest()
        else:
            tag_data = [
                '|'.join(forward_tags),
                '|'.join(backward_tags),
                klass,
                partner_code,
                with_baggage,
            ]
            tag_text = '{}@{}@{}@{}@{}'.format(*tag_data)
            return hashlib.md5(tag_text).hexdigest()

    def booking_available(self, now):
        if not self.forward.segments[0].departure:
            return True

        departure = self.forward.segments[0].departure

        hours = PARTNERS_BOOKING_AVAILABLE_HOURS.get(
            self.partner.code if self.partner else None, 0.75  # 45 min =>  3/4 hour => 0.75 hour
        )

        if departure and now > departure - timedelta(hours=hours):
            return False

        return True

    @property
    def baggage(self):
        return [
            [segment.baggage for segment in self.forward.segments],
            [segment.baggage for segment in self.backward.segments],
        ]

    @property
    def with_baggage(self):
        return all(chain.from_iterable(self.baggage))

    @property
    def selfconnect(self):
        return any(segment.selfconnect for segment in self.iter_all_segments())

    @property
    def is_direct(self):
        return len(self.forward.segments) == 1 and len(self.backward.segments) <= 1

    @property
    def completed_ok(self):
        return all(segment.is_complete for segment in self.iter_all_segments())

    def forward_exists(self):
        return bool(self.forward.segments)

    def forward_transfer_ok(self):
        return self.forward.check_transfers_times()

    def backward_fill_if_need(self, query):
        return not query.date_backward or bool(self.backward and self.backward.segments)

    def backward_transfer_ok(self):
        return not self.backward.segments or self.backward.check_transfers_times()

    def departure_after_arrival_at_different_steps(self):
        # is_arrival_on_previous_step_less_then_departure
        return not (
            self.backward.segments
            and self.backward.local_departure
            and self.forward.local_arrival
            and self.backward.local_departure - self.forward.local_arrival < MIN_STAY_TIME
        ) or not (
            self.backward.segments
            and self.backward.departure
            and self.forward.arrival
            and self.backward.departure - self.forward.arrival < MIN_STAY_TIME
        )

    def is_segment_service_ok(self, query):
        priorities = self.get_service_priorities()
        return not (priorities and min(priorities) < SERVICE_CLASS_PRIORITIES[query.klass])

    def variant_service_ok(self, query):
        return not (
            self.klass
            and self.klass in SERVICE_CLASS_PRIORITIES
            and query.klass in SERVICE_CLASS_PRIORITIES
            and SERVICE_CLASS_PRIORITIES[self.klass] < SERVICE_CLASS_PRIORITIES[query.klass]
        )

    def date_forward_ok(self, query):
        return not (
            self.forward.segments[0].local_departure
            and self.forward.segments[0].local_departure.date() != query.date_forward
        )

    def date_backward_ok(self, query):
        return not (
            self.backward.segments
            and self.backward.segments[0].local_departure
            and self.backward.segments[0].local_departure.date() != query.date_backward
        )

    def price_ok(self):
        return self.tariff.value > 0

    def _is_circle_route(self, station, visited_stations, visited_settlements):
        """
        Чтоб найти кольцевой перелет записываем посещенные города и станции и ищем ситуации:

        1. Дважды прибываем в одну станцию
        2. Повторяется город, но до этого мы были в другом городе (иначе это пересадка)
        """
        if station in visited_stations:
            return True

        current_settlement = get_airport_by_id(station)
        if current_settlement is None:
            visited_stations.append(station)
            return False

        current_settlement_id = current_settlement.settlement_id

        if current_settlement_id in visited_settlements:
            last_visited_settlement = get_airport_by_id(visited_stations[-1])
            if last_visited_settlement is not None and last_visited_settlement.settlement_id != current_settlement_id:
                return True

        visited_stations.append(station)
        visited_settlements.add(current_settlement_id)
        return False

    def _is_not_circle_route(self, segments):
        stations_from, stations_to = [], []
        settlements_from, settlements_to = set(), set()

        for segment in segments:
            if self._is_circle_route(segment.station_from.id, stations_from, settlements_from) or self._is_circle_route(
                segment.station_to.id, stations_to, settlements_to
            ):
                return False
        return True

    def is_not_circle_variant(self):
        return self._is_not_circle_route(self.forward.segments) and self._is_not_circle_route(self.backward.segments)

    def get_service_priorities(self):
        return filter(None, map(SERVICE_CLASS_PRIORITIES.get, self.klasses))

    def check_variant(self, rates, national_version, rules):
        # Хак. QueryBlackList пока есть только для Partner
        if isinstance(self.partner, Partner):
            rules = [r for r in rules if not r.partner or r.partner.code == self.partner.code]
        else:
            rules = [r for r in rules if not r.partner]

        if not rules:
            return False

        backward_segments = self.backward.segments
        forward_segments = self.forward.segments

        forward_station_from_id = forward_segments[0].station_from and forward_segments[0].station_from.id or None
        forward_station_to_id = forward_segments[-1].station_to and forward_segments[-1].station_to.id or None
        backward_station_from_id = (
            backward_segments and backward_segments[0].station_from and backward_segments[0].station_from.id or None
        )
        backward_station_to_id = (
            backward_segments and backward_segments[-1].station_to and backward_segments[-1].station_to.id or None
        )

        forward_settlement_from_id = forward_segments[0].station_from and forward_segments[0].station_from.settlement_id

        forward_settlement_to_id = forward_segments[-1].station_to and forward_segments[-1].station_to.settlement_id

        backward_settlement_from_id = (
            (backward_segments[0].station_from and backward_segments[0].station_from.settlement_id)
            if backward_segments
            else None
        )

        backward_settlement_to_id = (
            (backward_segments[-1].station_to and backward_segments[-1].station_to.settlement_id)
            if backward_segments
            else None
        )

        forward_country_from_id = _get_country_id_by_settlement_id(forward_settlement_from_id)
        forward_country_to_id = _get_country_id_by_settlement_id(forward_settlement_to_id)
        backward_country_from_id = _get_country_id_by_settlement_id(backward_settlement_from_id)
        backward_country_to_id = _get_country_id_by_settlement_id(backward_settlement_to_id)

        # Country
        rules = [
            r
            for r in rules
            if not r.country_from_id
               or (r.country_from_id == forward_country_from_id or r.country_from_id == backward_country_from_id)
        ]

        rules = [
            r
            for r in rules
            if not r.country_to_id
               or (r.country_to_id == forward_country_to_id or r.country_to_id == backward_country_to_id)
        ]

        # Settlement
        rules = [
            r
            for r in rules
            if not r.settlement_from_id
               or (
                   r.settlement_from_id == forward_settlement_from_id
                   or r.settlement_from_id == backward_settlement_from_id
               )
        ]

        rules = [
            r
            for r in rules
            if not r.settlement_to_id
               or (r.settlement_to_id == forward_settlement_to_id or r.settlement_to_id == backward_settlement_to_id)
        ]

        # Station
        rules = [
            r
            for r in rules
            if not r.station_from_id
               or (r.station_from_id == forward_station_from_id or r.station_from_id == backward_station_from_id)
        ]

        rules = [
            r
            for r in rules
            if not r.station_to_id
               or (r.station_to_id == forward_station_to_id or r.station_to_id == backward_station_to_id)
        ]

        if not rules:
            return False

        # Company
        segment_company_ids = {segment.company.id for segment in self.iter_all_segments() if segment.company}
        rules = [r for r in rules if not r.company_id or r.company_id in segment_company_ids]

        # Flight number
        segments_flight_numbers = {str(s.number.split()[-1]) for s in self.iter_all_segments()}
        rules = [r for r in rules if not r.flight_number or str(r.flight_number) in segments_flight_numbers]

        # National version
        rules = [r for r in rules if not r.national_version or r.national_version == national_version]

        if not rules:
            return False

        # Currency
        rules_by_currency = {}
        for r in rules:
            rules_by_currency.setdefault(r.currency_id or None, []).append(r)
        rules = rules_by_currency.pop(None, [])

        for currency_id, one_currency_rules in rules_by_currency.iteritems():
            rule_currency = get_currency_by_id(currency_id)

            if self.tariff.currency == rule_currency.code:
                price = self.tariff.value
            else:
                if not rates or self.tariff.currency not in rates or rule_currency.code not in rates:
                    continue
                price = float(self.tariff.value) * float(rates[self.tariff.currency]) / float(rates[rule_currency.code])

            rules.extend(
                r
                for r in one_currency_rules
                if ((not r.price_from or float(r.price_from) < price) and (not r.price_to or float(r.price_to) > price))
            )

        if not rules:
            return False

        # When
        if forward_segments[0].local_departure:
            departure_date = forward_segments[0].local_departure.date()
            rules = [r for r in rules if (not r.when_from or r.when_from <= departure_date)]

        if forward_segments[-1].local_arrival:
            arrival_date = forward_segments[-1].local_arrival.date()
            rules = [r for r in rules if (not r.when_to or r.when_to >= arrival_date)]

        # Service class
        rules = [r for r in rules if not r.klass or r.klass == self.klass]

        return bool(rules)

    @property
    def all_segments(self):
        if self.backward.segments:
            return self.forward.segments + self.backward.segments
        else:
            return self.forward.segments

    def iter_all_segments(self):
        return chain(self.forward.segments, self.backward.segments)

    @property
    def klasses(self):
        return self.get_klasses()

    def get_klasses(self):
        return {s.klass for s in self.iter_all_segments() if s.klass}

    @property
    def is_charter(self):
        # typing: () -> bool
        return bool(self.charter)

    @property
    def by_flights_key(self):
        return tuple((f.number, f.local_departure) for f in self.all_segments)

    @property
    def by_fare_code(self):
        if feature_flags.store_min_tariff_per_fare_code():
            return tuple(f.fare_code if f.fare_code else '' for f in self.iter_all_segments())
        else:
            return tuple('' for f in self.iter_all_segments())

    def __str__(self):
        try:
            return '%s[%s](%s) %s [%s]' % (
                self.__class__.__name__,
                getattr(self, 'partner', None),
                self.tariff,
                self.klass,
                self.by_flights_key,
            )
        except Exception:
            log.exception('Flight.__str__')
            return repr(self)


@memoize(lambda avia_company, fare_code: (avia_company.pk, fare_code))
def get_aviacompany_tariff_by_fare_code(avia_company, fare_code):
    tariffs_by_aviacompany_pk = get_tariffs_by_aviacompany_pk()

    if avia_company.pk not in tariffs_by_aviacompany_pk:
        return None
    tariffs = tariffs_by_aviacompany_pk[avia_company.pk]

    default_tariff = None
    for company_tariff in tariffs:
        if company_tariff.mask:
            if company_tariff.mask_match_code(fare_code or ''):
                return company_tariff
        else:
            default_tariff = company_tariff
    return default_tariff
