# -*- coding: utf-8 -*-
from collections import defaultdict
from operator import attrgetter
from django.db.models import Q

from travel.avia.library.python.avia_data.models.company import AviaCompany, CompanyTariff
from travel.avia.library.python.common.models.iatacorrection import IataCorrection
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.utils.iterrecipes import group_by

from travel.avia.library.python.ticket_daemon.memo import memoize, SimpleWarmGroup


company_warm_group = SimpleWarmGroup('company')


@company_warm_group
@memoize()
def _aviacompanies_by_company_id():
    return {c.rasp_company_id: c for c in AviaCompany.objects.all()}


def get_aviacompany_by_company_id(company_id):
    return _aviacompanies_by_company_id().get(company_id)


def make_company_by_id():
    return _company_with_codes()


@company_warm_group
@memoize()
def _company_with_codes():
    companies = (
        Company.objects.filter(
            Q(t_type_id=TransportType.PLANE_ID) |
            Q(iata__isnull=False) |
            Q(sirena_id__isnull=False) |
            Q(icao__isnull=False)
        )
    )
    return {c.id: c for c in companies}


def get_company_by_id(company_id):
    return make_company_by_id()[company_id]


def get_companies_by_iata(iata):
    return make_companies_by_iata().get(iata, [])


@company_warm_group
@memoize()
def _iatacorrections():
    return list(IataCorrection.objects.all())


@company_warm_group
@memoize()
def make_corrections_by_iata():
    return dict(group_by(_iatacorrections(), attrgetter('code')))


@company_warm_group
@memoize()
def make_companies_by_iata():
    companies_by_id = make_company_by_id()
    companies = companies_by_id.itervalues()

    return {
        code: sorted(cs, key=attrgetter('priority'), reverse=True)
        for code, cs in group_by(companies, attrgetter('iata')) if code
    }


@company_warm_group
@memoize()
def make_companies_by_sirena():
    companies_by_id = make_company_by_id()
    companies = companies_by_id.itervalues()

    return {
        code: sorted(cs, key=attrgetter('priority'), reverse=True)
        for code, cs in group_by(companies, attrgetter('sirena_id')) if code
    }


# RASPTICKETS-16594 в этой задаче надо проверить работает ли это и раскоментить
# @company_warm_group
# @memoize()
# def make_companies_by_icao():
#     companies_by_id = make_company_by_id()
#     companies = companies_by_id.itervalues()
#
#     return {
#         code: sorted(cs, key=attrgetter('priority'), reverse=True)
#         for code, cs in group_by(companies, attrgetter('icao')) if code
#     }


# @company_warm_group
# @memoize()
# def make_companies_by_icao_ru():
#     companies_by_id = make_company_by_id()
#     companies = companies_by_id.itervalues()
#
#     return {
#         code: sorted(cs, key=attrgetter('priority'), reverse=True)
#         for code, cs in group_by(companies, attrgetter('icao_ru')) if code
#     }


@company_warm_group
@memoize()
def get_tariffs_by_aviacompany_pk():
    by_aviacompany_pk = defaultdict(list)
    for t in CompanyTariff.objects.all():
        by_aviacompany_pk[t.avia_company_id].append(t)
    return dict(by_aviacompany_pk)
