# -*- coding: utf-8 -*-
from itertools import chain
from typing import Dict, List, Optional, Union

import opentracing

from travel.avia.library.python.avia_data.models import AmadeusMerchant
from travel.avia.library.python.common.models.partner import Partner, DohopVendor
from travel.avia.library.python.ticket_daemon.memo import SimpleWarmGroup, memoize, CacheWithKeyTTL


def _partners():
    # type: () -> List[Union[Partner, DohopVendor, AmadeusMerchant]]

    return list(
        chain(
            Partner.objects.all(),
            DohopVendor.objects.all(),
            AmadeusMerchant.objects.all(),
        )
    )


@SimpleWarmGroup('partners')
@memoize()
def _partner_by_code():
    # type: () -> Dict[str, Union[Partner, DohopVendor, AmadeusMerchant]]

    return {p.code: p for p in _partners()}


def get_partner_by_code(code):
    # type: (str) -> Optional[Union[Partner, DohopVendor, AmadeusMerchant]]

    return _partner_by_code().get(code)


def _gap_cache_key(national_version, t_code='plane', from_rasp=False, mobile=False):
    return national_version, t_code, from_rasp, mobile


def get_actual_partners_without_cache(national_version, t_code, from_rasp, mobile):
    # type: (str, str, bool, bool) -> Union[Partner, AmadeusMerchant, DohopVendor]

    with opentracing.global_tracer().start_active_span(operation_name='Getting actual partners', finish_on_close=True):
        with opentracing.global_tracer().start_active_span(
            operation_name='Getting regular partners', finish_on_close=True
        ):
            partners = Partner.get_actual(t_code, national_version, from_rasp=from_rasp, mobile=mobile)

        # Если Дохоп в списке, то добавим и его вендоров
        if any(p.code == 'dohop' for p in partners):
            with opentracing.global_tracer().start_active_span(
                operation_name='Getting Dohop vendors', finish_on_close=True
            ):
                partners = [p for p in partners if p.code != 'dohop']
                partners += DohopVendor.get_actual(national_version, mobile=mobile)

        # Если Амадеус в списке, то добавим и его AmadeusMerchant
        if any(p.code == 'amadeus' for p in partners):
            with opentracing.global_tracer().start_active_span(
                operation_name='Getting Amadeus merchants', finish_on_close=True
            ):
                partners = [p for p in partners if p.code != 'amadeus']
            partners += AmadeusMerchant.get_actual(national_version, mobile=mobile)

        return partners


TTL = 60

partners_cache = CacheWithKeyTTL(TTL)


@memoize(_gap_cache_key, partners_cache)
def get_actual_partners(national_version, t_code='plane', from_rasp=False, mobile=False):
    return get_actual_partners_without_cache(
        national_version=national_version, t_code=t_code, from_rasp=from_rasp, mobile=mobile
    )
