# -*- coding: utf-8 -*-
from collections import defaultdict

from travel.avia.library.python.common.models.geo import SettlementCode
from travel.avia.library.python.ticket_daemon.memo import memoize, SimpleWarmGroup

from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.station import (
    _airport_by_id, _station2settlement, _settlement_ids_by_airport_id
)
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.settlement import _settlement_by_id


@SimpleWarmGroup('search_codes')
@memoize()
def _iata_codes_for_search_by_point_key():
    """
    В iata записывается:
    аэропорт ->
        first(код аэропорта, код города аэропорта)
    город ->
        first(код города если в нём есть аэропорты ,
              коды аэропортов города) ,
        first(берутся s2s аэропорты, выбираются непустые коды аэропортов ,
              берутся s2s аэропорты, выбираются непустые коды городов) ,

    :rtype: dict[unicode, list[unicode]]
    """

    airport_by_id = _airport_by_id()
    airports = airport_by_id.values()
    airports_by_settlement_id = defaultdict(list)

    for a in airports:
        if a.settlement_id and a.iata:
            airports_by_settlement_id[a.settlement_id].append(a)

    airport_iatas_by_settlement_id = {
        settlement_id: [a.iata for a in sorted(airports, key=lambda a: a.hidden)]
        for settlement_id, airports in airports_by_settlement_id.iteritems()
    }

    s2s_airports_by_settlement_id = defaultdict(set)
    for s2s in _station2settlement():
        airport = airport_by_id.get(s2s.station_id)
        if airport and airport.iata:
            s2s_airports_by_settlement_id[s2s.settlement_id].add(airport)

    settlement_by_id = _settlement_by_id()

    def get_settlement_iata_by_settlement_id(settlement_id):
        settlement = settlement_by_id.get(settlement_id)
        return settlement.iata if settlement else None

    def get_settlement_related_iata_by_settlement_id(settlement_id):
        settlement_airport_iatas = airport_iatas_by_settlement_id.get(settlement_id)
        if settlement_airport_iatas:
            settlement = settlement_by_id.get(settlement_id)
            if settlement and settlement.iata:
                # код города если в нём есть аэропорты
                yield settlement.iata
            else:
                # на случай если город обзавёлся несколькими аэропортами, но до сих пор не имеет своего
                # iata кода
                for settlement_airport_iata in settlement_airport_iatas:
                    yield settlement_airport_iata
        # берутся s2s аэропорты с не пустым кодом город аэропорта
        s2airports = s2s_airports_by_settlement_id.get(settlement_id, [])
        for aprt in s2airports:
            settlement_iata = get_settlement_iata_by_settlement_id(aprt.settlement_id)
            if settlement_iata:
                yield settlement_iata
            else:
                yield aprt.iata
        return

    by_point_key = defaultdict(list)

    for a in airports:
        pkey = a.point_key
        code = a.iata or get_settlement_iata_by_settlement_id(a.settlement_id)
        if code:
            by_point_key[pkey] = [code]

    for s in _settlement_by_id().itervalues():
        pkey = s.point_key
        for code in get_settlement_related_iata_by_settlement_id(s.id):
            if code and code not in by_point_key[pkey]:
                by_point_key[pkey].append(code)

    return by_point_key


def get_iata_code_for_search(point_key):
    """
    :param unicode point_key:
    :return: first related iata code
    :rtype: unicode
    """
    codes = get_iata_codes_for_search(point_key)
    if codes:
        return next(iter(codes), None)
    else:
        return None


def get_iata_codes_for_search(point_key):
    """
    :param unicode point_key:
    :return: iterator over iata codes
    :rtype: typing.Iterable[unicode]
    """

    return _iata_codes_for_search_by_point_key().get(point_key, None)


@memoize()
def _sirena_points_for_search_by_point_key():
    """
    Поиск аналогично iata кодам

    :rtype: dict[unicode, list[unicode]]
    """

    airport_by_id = _airport_by_id()
    airports = airport_by_id.values()
    airports_by_settlement_id = defaultdict(list)

    for a in airports:
        if a.settlement_id and a.sirena_id:
            airports_by_settlement_id[a.settlement_id].append(a)

    airport_sirenas_by_settlement_id = {
        settlement_id: [a.sirena_id for a in sorted(airports, key=lambda a: a.hidden)]
        for settlement_id, airports in airports_by_settlement_id.iteritems()
    }

    s2s_airports_by_settlement_id = defaultdict(set)
    for s2s in _station2settlement():
        airport = airport_by_id.get(s2s.station_id)
        if airport and airport.sirena_id:
            s2s_airports_by_settlement_id[s2s.settlement_id].add(airport)

    settlement_by_id = _settlement_by_id()

    def get_settlement_sirena_by_settlement_id(settlement_id):
        settlement = settlement_by_id.get(settlement_id)
        return settlement.sirena_id if settlement else None

    def get_settlement_related_sirena_by_settlement_id(settlement_id):
        settlement_airport_sirenas = airport_sirenas_by_settlement_id.get(settlement_id)
        if settlement_airport_sirenas:
            settlement = settlement_by_id.get(settlement_id)
            if settlement and settlement.sirena_id:
                # город если в нём есть аэропорты с сирена—кодом
                yield settlement.sirena_id
            else:
                for settlement_sirena_airport in settlement_airport_sirenas:
                    yield settlement_sirena_airport
        # берутся s2s аэропорты с не пустым кодом город аэропорта
        s2airports = s2s_airports_by_settlement_id.get(settlement_id, [])
        for aprt in s2airports:
            settlement_sirena = get_settlement_sirena_by_settlement_id(aprt.settlement_id)
            if settlement_sirena:
                yield settlement_sirena
            else:
                yield aprt.sirena_id
        return

    by_point_key = defaultdict(list)

    for a in airports:
        pkey = a.point_key
        code = a.sirena_id or get_settlement_sirena_by_settlement_id(a.settlement_id)
        if code:
            by_point_key[pkey] = [code]

    for s in _settlement_by_id().itervalues():
        pkey = s.point_key
        for code in get_settlement_related_sirena_by_settlement_id(s.id):
            if code and code not in by_point_key[pkey]:
                by_point_key[pkey].append(code)

    return by_point_key


def get_sirena_code_for_search(point_key):
    """
    :param unicode point_key:
    :rtype: unicode
    """
    codes = get_sirena_codes_for_search(point_key)
    if codes:
        return next(iter(codes), None)
    else:
        return None


def get_sirena_codes_for_search(point_key):
    """
    :param unicode point_key:
    :rtype: typing.Iterable[unicode]
    """
    return _sirena_points_for_search_by_point_key().get(point_key)


@SimpleWarmGroup('chartex_citycodes')
@memoize()
def _chartex_city_code_by_settlement_id():
    return dict(SettlementCode.objects.filter(system__code='chartex')
                .namedtuples('settlement_id', 'code'))


@SimpleWarmGroup('chartex_citycodes')
@memoize()
def _chartex_city_code_by_point_key():
    chartex_city_code_by_settlement_id = dict(
        SettlementCode.objects.filter(system__code='chartex')
            .namedtuples('settlement_id', 'code'))

    by_point_key = {}
    for settlement in _settlement_by_id().values():
        code = chartex_city_code_by_settlement_id.get(settlement.id)
        if code:
            by_point_key[settlement.point_key] = code

    for airport in _airport_by_id().itervalues():
        # Сначала будут id городов, привязанных непосредственно,
        # а после — через station2settlement
        settlement_ids_by_airport_id = _settlement_ids_by_airport_id()
        settlement_ids = settlement_ids_by_airport_id.get(airport.id, [])
        for settlement_id in settlement_ids:
            code = chartex_city_code_by_settlement_id.get(settlement_id)
            if code:
                by_point_key[airport.point_key] = code
                break

    return by_point_key


def get_chartex_city_code_by_point(point):
    return _chartex_city_code_by_point_key().get(point.point_key)
