# -*- coding: utf-8 -*-
from collections import defaultdict, OrderedDict
from itertools import chain

from django.db.models import Q

from travel.avia.library.python.common.models.geo import Station, StationCode, Station2Settlement
from travel.avia.library.python.common.models.transport import TransportType

from travel.avia.library.python.ticket_daemon.memo import memoize, SimpleWarmGroup


station_codes_warm_group = SimpleWarmGroup('station_codes')
airport_warm_group = SimpleWarmGroup('airport')


@station_codes_warm_group
@memoize()
def _stations_iatas_by_station_id():
    return dict(StationCode.objects.filter(system__code='iata')
                .values_list('station_id', 'code'))


@station_codes_warm_group
@memoize()
def _station_id_by_iata():
    return {iata: sid for sid, iata in _stations_iatas_by_station_id().iteritems()}


def station_iata(station):
    return _stations_iatas_by_station_id().get(station.id)


def monkey_patch_station_iata():
    """
    Либо нужно во всех местах где используется station.iata, писать
    station_iata(station)
    """
    def iata(self):
        return station_iata(self)
    Station.iata = property(iata)


@station_codes_warm_group
@memoize()
def _stations_sirenas_by_station_id():
    return dict(StationCode.objects.filter(system__code='sirena')
                .values_list('station_id', 'code'))


@station_codes_warm_group
@memoize()
def _station_id_by_sirena():
    return {sirena: sid for sid, sirena in _stations_sirenas_by_station_id().iteritems()}


@memoize(lambda system_code: system_code)
def _station_id_by_codesystem(system_code):
    return dict(StationCode.objects.filter(system__code=system_code)
                .values_list('code', 'station_id'))


@memoize(lambda code: code)
def get_station_by_iata_or_sirena(code):
    station_id = (_station_id_by_iata().get(code) or
                  _station_id_by_sirena().get(code))
    if station_id:
        return _airport_by_id().get(station_id)


@airport_warm_group
@memoize()
def _airport_by_id():
    """
    Пришлось импортировать здесь для обхода циклических импортов. В таких
    случаях можно руководствоваться одинм принципом: Интерфейсы импортировать
    по месту, а из интерфейсов импортировать всё что нужно в начале модуля.

    :rtype: dict[int, Station]
    """
    from .interfaces import StationInterface
    station_ids = set(chain(
        _station_id_by_iata().itervalues(),
        _station_id_by_sirena().itervalues()
    ))
    iata_by_station_id = _stations_iatas_by_station_id()

    # hidden=False  # Разрешаем пока все, даже hidden
    qs = Q(id__in=station_ids) | Q(t_type_id=TransportType.PLANE_ID)

    return {
        s.id: s for s in Station.objects.filter(qs).namedtuples(
            'id',
            'settlement_id',
            'region_id',
            'country_id',
            'time_zone',
            'sirena_id',
            'title',
            'hidden',
            computational={
                'point_key': (lambda row: 's{}'.format(row[0])),
                'iata': (lambda row: iata_by_station_id.get(row[0])),
            },
            interface=StationInterface
        )
    }


def get_airport_by_id(station_id):
    return _airport_by_id().get(station_id)


@SimpleWarmGroup('station2settlement')
@memoize()
def _station2settlement():
    return list(
        Station2Settlement.objects.
        filter(station_id__in=_airport_by_id().keys()).
        namedtuples('station_id', 'settlement_id')
    )


class OrderedSet(OrderedDict):
    """
    В итоге чтобы получить значения, нужно будет преобразовать в список
    """
    def add(self, value):
        if value not in self:
            self[value] = None


@airport_warm_group
@memoize()
def _airport_ids_by_settlement_id():
    by_settlement = defaultdict(OrderedSet)

    for airport in _airport_by_id().values():
        if airport.settlement_id:
            by_settlement[airport.settlement_id].add(airport.id)

    for s2s in _station2settlement():
        by_settlement[s2s.settlement_id].add(s2s.station_id)

    return {
        settlement_id: tuple(stations_ids)
        for settlement_id, stations_ids in by_settlement.iteritems()
    }


def get_airport_ids_by_settlement_id(settlement_id):
    return _airport_ids_by_settlement_id().get(settlement_id)


@airport_warm_group
@memoize()
def _settlement_ids_by_airport_id():
    by_airport = defaultdict(OrderedSet)

    # Сначала города, привязанные явно
    for airport in _airport_by_id().values():
        if airport.settlement_id:
            by_airport[airport.id].add(airport.settlement_id)

    # После — через station2settlement
    for s2s in _station2settlement():
        by_airport[s2s.station_id].add(s2s.settlement_id)

    return {
        settlement_id: tuple(stations_ids)
        for settlement_id, stations_ids in by_airport.iteritems()
    }


def get_settlement_ids_by_airport_id(settlement_id):
    return _settlement_ids_by_airport_id().get(settlement_id)
