# -*- coding: utf-8 -*-
from django.conf import settings
from logging import getLogger

from travel.avia.ticket_daemon.ticket_daemon.api.partners_utils import get_partner_by_code
from travel.avia.ticket_daemon.ticket_daemon.api.query import get_query_module
from travel.avia.ticket_daemon.ticket_daemon.api.yaclid import YaClid
from travel.avia.ticket_daemon.ticket_daemon.lib.http import dirty_add_to_query_string
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.partners import add_marker_to_url, book, generate_marker


log = getLogger(__name__)
DAEMON_API_REDIRECT_HTTP_TIMEOUT = 5
DAEMON_API_REDIRECT_HTTP_PARTNERS_TIMEOUT = {
    'pilotua': 10,
    'dohop': 15,
}


class RedirectError(Exception):
    pass


class AdditionalDataRequired(Exception):
    pass


def get_module_function(partner, func_name, default_func, log_prefix):
    query_module = get_query_module(partner)

    if not query_module:
        raise RuntimeError(u'Query module not found for partner %r' % partner)

    try:
        module_function = getattr(query_module, func_name)
    except Exception:
        module_function = default_func

    log.debug('%s: %r' % (log_prefix, FunctionRepr(module_function)))

    return module_function


class FunctionRepr(object):
    def __init__(self, fn):
        self.fn = fn

    def __repr__(self):
        return '%s:%s' % (self.fn.__module__, self.fn.__name__)


def format_redirector_result(redir):
    if isinstance(redir, basestring):
        return {'url': redir}

    elif isinstance(redir, tuple) and len(redir) == 2:
        url, post_data = redir
        return {'url': url, 'post_data': post_data}

    elif isinstance(redir, dict):
        return slice_dict(redir, ('url', 'm_url', 'post_data', 'marker'))


def process_redirector_result(partner, redirect_data):
    marker = get_module_function(partner, 'generate_marker', generate_marker, 'Marker generator')()
    marker_field = partner.marker or 'marker'

    use_post_data = 'post_data' in redirect_data

    partner_is_for_backend_generation = partner.code in settings.BACKEND_MARKER_GENERATION_PARTNER_NAMES
    use_new_marker_generation = settings.USE_BACKEND_MARKER_GENERATION or partner_is_for_backend_generation

    if use_new_marker_generation:
        if use_post_data:
            redirect_data['post_data'][marker_field] = marker
        redirect_data['marker'] = marker

    for url_key in ['url', 'm_url']:
        modify_key(redirect_data, url_key, add_yaclid)
        if use_new_marker_generation:
            marker_adder = get_module_function(partner, 'add_marker_to_url', add_marker_to_url, 'Marker adder')
            modify_key(redirect_data, url_key, lambda url: marker_adder(marker_field, marker, url))

    return redirect_data


def fetch_redirect_data(partner, order_data):
    log.info('start')

    redirect_data = get_module_function(partner, 'book', book, 'Redirector')(order_data)
    redirect_data = format_redirector_result(redirect_data)
    if not redirect_data:
        return None

    redirect_data = process_redirector_result(partner, redirect_data)

    log.info('redirect_data: %r', slice_dict(redirect_data, ['url', 'm_url']))
    return redirect_data


def slice_dict(d, keys):
    return {k: d[k] for k in keys if k in d}


def modify_key(d, key, fn):
    if key in d:
        d[key] = fn(d[key])


def add_yaclid(url):
    if not url:
        return url
    return dirty_add_to_query_string(url, {
        'yaclid': YaClid.create().dumps(),
    })


class RedirectData(object):
    def __init__(self, url, post=None, m_url=None, query_source=None):
        self.url = url
        self.post = post
        self.m_url = m_url
        self.query_source = query_source

    def __json__(self):
        return skip_None_values({
            'url': self.url,
            'post': self.post,
        })

    def redirect_url(self, is_mobile=False):
        if is_mobile:
            return self.m_url or self.url
        else:
            return self.url


def get_redirect_partner(order_data):
    partner = get_partner_by_code(order_data['partner'])

    if not partner:
        raise RedirectError('No partner')

    if not partner.enabled:
        raise RedirectError('Not partner.enabled [%s]', partner.code)

    return partner


def get_partner_redirect_timeout(partner_code):
    if partner_code.startswith('dohop_'):
        partner_code = 'dohop'
    return DAEMON_API_REDIRECT_HTTP_PARTNERS_TIMEOUT.get(
        partner_code,
        DAEMON_API_REDIRECT_HTTP_TIMEOUT
    )
