# -*- coding: utf-8 -*-
import logging
from functools import partial

from datetime import datetime

from django.conf import settings

from travel.avia.library.python.common.models.partner import Partner

from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.regionalization_rules import get_partner_rules
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import group_by

log = logging.getLogger(__name__)


def apply(query, partners):
    _can_ask_partner = partial(can_ask_partner, query)
    can_ask_partner_groups = group_by(partners, _can_ask_partner)
    return can_ask_partner_groups[True], can_ask_partner_groups[False]


def can_ask_partner(query, partner):
    # Пока работает только для обычного партнёра
    if not isinstance(partner, Partner):
        return True

    if partner.code in settings.PARTNERS_DONT_QUERY_TODAY:
        now_date = datetime.now().date()

        if now_date >= query.date_forward:
            log.info(u'Не спрашиваем %s с вылетом ранее чем завтра. query:%r', partner.code, query.id)
            return False

    value = _find_out_can_ask_partner(query, partner)

    if value:
        log.info(u'Спрашиваем %s по правилам регионализации. query:%r', partner.code, query.id)

    else:
        log.info(u'Не спрашиваем %s по правилам регионализации. query:%r', partner.code, query.id)

    return value


def _find_out_can_ask_partner(query, partner):
    regionalization_rules = get_partner_rules(partner.code)

    if regionalization_rules is None:
        return True

    applicable_rules = regionalization_rules.get_applicable_rules(query)
    if not applicable_rules and regionalization_rules.only_exclude_rules:
        return True  # Разрешаем спрашивать только если все несовпавшие правила - "кроме"

    for rule in applicable_rules:
        if rule.exclude:
            if _is_rule_applicable_by_departure_date(rule, query):
                log.info(u'Blocked with rule: %r %s. query:%r', rule, partner.code, query.id)
                return False

    for rule in applicable_rules:
        if not rule.exclude:
            if _is_rule_applicable_by_departure_date(rule, query):
                log.info(u'Allowed with rule: %r %s. query:%r', rule, partner.code, query.id)
                return True

    return False


def _is_rule_applicable_by_departure_date(rule, query):
    """Проверяем, что дата вылета и день недели запроса попадает в ограничения правила"""
    if rule.start_date is not None and rule.end_date is not None:
        departure_inside = rule.start_date <= query.date_forward <= rule.end_date
        if rule.week_days is not None:
            return departure_inside and str(query.date_forward.weekday() + 1) in rule.week_days
        return departure_inside
    return True
