# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals

import gevent
import json
import zlib

from django.conf import settings
from logging import getLogger

from travel.avia.ticket_daemon.ticket_daemon.lib.http import with_retry


logger = getLogger(__name__)

# Makes zlib.compress() use gzip header and footer. See: https://docs.python.org/3/library/zlib.html#zlib.compressobj
WBITS_WITH_GZIP_HEADER = 31


# Fetch fare data from the fare-families service and add that data to the result
class FareFamiliesProvider(object):

    @staticmethod
    def get_fare_families_for_variants(serialized_variants):
        fare_families = FareFamiliesProvider._fetch_fare_families(serialized_variants)
        logger.debug('fetched fare-families: %s', fare_families)
        return fare_families

    @staticmethod
    def _fetch_fare_families(serialized_variants):
        session = with_retry(backoff_factor=0.2)
        try:
            # the definition of the response payload is located here:
            # https://a.yandex-team.ru/arc/trunk/arcadia/travel/avia/fare_families/internal/services/fare_families/data_structs/payloads/tariff_payload.go?rev=r9107943#L72
            compressor = zlib.compressobj(zlib.Z_DEFAULT_COMPRESSION, zlib.DEFLATED, WBITS_WITH_GZIP_HEADER)
            compressed_data = compressor.compress(json.dumps(dict(variants=[serialized_variants]))) + compressor.flush()

            response = gevent.with_timeout(
                settings.FARE_FAMILIES_TIMEOUT*3,
                session.post,
                url=settings.FARE_FAMILIES_URL,
                data=compressed_data,
                headers={
                    'Content-Encoding': 'gzip',
                    'Accept-Encoding': 'gzip',
                },
                timeout=settings.FARE_FAMILIES_TIMEOUT,
            )

            if response.ok:
                return response.json()
            else:
                logger.error(
                    'Fare families response is faulty, status=%s, content=%s', response.status_code, response.content)
                return None

        except gevent.Timeout:
            logger.error('Timeout on a fare families request')
            return None

        except Exception as e:
            logger.exception('Error %s on a fare families request', e)
            return None
