# -*- coding: utf-8 -*-
import logging
from itertools import chain
from operator import attrgetter
from datetime import datetime
from typing import Dict, List, Optional
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, OperatingFlight, IATAFlight

import pytz

from travel.avia.library.python.ticket_daemon.date import DateTimeSerializerV3, EPOCH

logger = logging.getLogger(__name__)

DT_FORMAT = '%Y-%m-%dT%H:%M:%S'


def _model_to_serialize(model):
    return model and model.id


class FlightsSerializer(object):
    @staticmethod
    def serialize(variants):
        # type: (List[Variant])->Dict[str, Dict]
        flights = {}
        _datetime_serializer = DateTimeSerializerV3()
        for v in variants:
            for flight in chain(v.forward.segments, v.backward.segments):  # type: IATAFlight
                if flight.key in flights:
                    continue

                flights[flight.key] = {
                    'from': _model_to_serialize(flight.station_from),
                    'to': _model_to_serialize(flight.station_to),
                    'companyTariff': _model_to_serialize(flight.company_tariff),
                    'key': flight.key,
                    'company': _model_to_serialize(flight.company),
                    'aviaCompany': flight.avia_company and flight.avia_company.rasp_company_id,
                    'number': flight.number,
                    'operating': FlightsSerializer._serialize_operating(flight.operating),
                    'departure': _datetime_serializer.serialize(
                        flight.departure) if flight.departure else None,
                    'arrival': _datetime_serializer.serialize(
                        flight.arrival) if flight.arrival else None,
                }
        return flights

    @staticmethod
    def _serialize_operating(operating_flight):
        # type: (OperatingFlight)->Optional[dict]
        if not operating_flight or not operating_flight.check_correctness(logging.ERROR):
            return None
        else:
            return {
                'company': _model_to_serialize(operating_flight.company),
                'number': operating_flight.number,
            }


class FaresSerializer(object):
    def serialize(self, variants, created_at, expires_at):
        """

        :type variants: list of ticket_daemon.api.flights.Variant
        :type created_at: int
        :type expires_at: int
        :rtype: dict
        """
        return {
            v.tag: self._serialize_fare(v, expires_at) for v in variants
        }

    def _serialize_fare(self, variant, expires_at):
        result = {
            'selfconnect': variant.selfconnect,
            'charter': variant.charter,
            'tariff': {
                'value': variant.tariff.value,
                'currency': variant.tariff.currency,
            },
            'baggage': self._baggage_keys(variant),
            'fare_codes': self._fare_codes(variant),
            'route': self._route_keys(variant),
            'created': variant.created_at,
            'expire': expires_at,
            'price_category': variant.price_category,
        }
        if 'promo' in variant.order_data:
            result['promo'] = variant.order_data['promo']
        if 'yandex_plus_promo' in variant.order_data:
            result['yandex_plus_promo'] = variant.order_data['yandex_plus_promo']
        return result

    def _route_keys(self, variant):
        """
        :type variant: ticket_daemon.api.flights.Variant
        :rtype: list of lists
        """
        return (
            tuple(map(attrgetter('key'), variant.forward.segments)),
            tuple(map(attrgetter('key'), variant.backward.segments)),
        )

    def _fare_codes(self, variant):
        """
            :type variant: ticket_daemon.api.flights.Variant
            :rtype: list of lists
        """
        return (
            tuple(map(attrgetter('fare_code'), variant.forward.segments)),
            tuple(map(attrgetter('fare_code'), variant.backward.segments)),
        )

    def _baggage_keys(self, variant):
        """

        :type variant: ticket_daemon.api.flights.Variant
        :rtype: list of lists
        """
        return [
            [b.key() for b in direction_baggage if b is not None]
            for direction_baggage in variant.baggage
        ]


class BigWizardFaresSerializer(FaresSerializer):
    def serialize(self, variants, created_at, expires_at):
        """
        Сохраняем только предложения, переданные в валюте национальной версии.
        :type variants: list of ticket_daemon.api.flights.Variant
        :type created_at: int
        :type expires_at: int
        :rtype: list
        """
        return [
            self._serialize_vbb_fare(v, created_at, expires_at) for v in variants
            if v.national_tariff == v.tariff
        ]

    def _serialize_vbb_fare(self, variant, created_at, expires_at):
        return {
            'charter': variant.charter,
            'tariff': {
                'value': variant.national_tariff.value,
                'currency': variant.national_tariff.currency,
            },
            'baggage': self._baggage_keys(variant),
            'route': self._route_keys(variant),
            'partner': variant.partner.code,
            'created': created_at,
            'expire': expires_at,
            'tariffs': {},
            'promo': self._promo(variant),
        }

    def _promo(self, variant):
        promo = variant.order_data.get('promo', {})
        if not promo or not promo.get('code'):
            return {}

        result = {'code': promo['code']}

        tz_code = promo.get('endActionTz')
        date_str = promo.get('endActionDt')
        if tz_code and date_str:
            tz = pytz.timezone(tz_code)
            date = datetime.strptime(date_str, DT_FORMAT)

            result['end_ts'] = int((tz.localize(date) - tz.localize(EPOCH)).total_seconds())
        return result

    def tariff_info(self, variant):
        return {
            'price': variant['tariff'],
            'partner': variant['partner'],
            'baggage': variant['baggage'],
            'created_at': variant['created'],
            'expire_at': variant['expire'],
        }


flights_serializer = FlightsSerializer()
fares_serializer = FaresSerializer()
big_wizard_fares_serializer = BigWizardFaresSerializer()
