# -*- coding: utf-8 -*-
from collections import defaultdict, namedtuple

from travel.avia.library.python.django_namedtuples.queryset import ModelInterface

from travel.avia.library.python.avia_data.models.selfbook import (
    SelfBookCompany, SelfBookDirection, SelfBookPartner,
    SelfBookNationalVersion, SelfBookRule,
)

from travel.avia.library.python.ticket_daemon.memo import memoize, CacheWithKeyTTL
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.point import get_point_by_iata_or_sirena


class SelfBookRuleInterface(ModelInterface):
    _fields = (
        'id',
        'exclude_partners',
        'exclude_companies',
        'exclude_directions',
    )

    def get_national_version_codes(self):
        return _rule_national_version_codes().get(self.id, [])

    def get_partner_codes(self):
        return _rule_partner_codes().get(self.id, [])

    def get_company_codes(self):
        return _rule_company_codes().get(self.id, [])

    def get_directions(self):
        return _rule_directions().get(self.id, [])


def agg_with_list(pairs):
    agg = defaultdict(list)
    for key, value in pairs:
        agg[key].append(value)
    return dict(agg)


@memoize(cache=CacheWithKeyTTL(30))
def _rule_national_version_codes():
    return agg_with_list(SelfBookNationalVersion.objects
                         .values_list('rule_id', 'national_version__code'))


@memoize(cache=CacheWithKeyTTL(30))
def _rule_partner_codes():
    return agg_with_list(SelfBookPartner.objects
                         .values_list('rule_id', 'partner__code'))


@memoize(cache=CacheWithKeyTTL(30))
def _rule_company_codes():
    return agg_with_list(SelfBookCompany.objects
                         .values_list('rule_id', 'company__iata'))


@memoize(cache=CacheWithKeyTTL(30))
def _rule_directions():
    return agg_with_list(
        (d.rule_id, d) for d in SelfBookDirection.objects.all().namedtuples(interface=SelfBookDirectionInterface)
    )


class SelfBookDirectionInterface(ModelInterface):
    _fields = (
        'rule_id',
        'settlement_from_id', 'country_from_id',
        'settlement_to_id', 'country_to_id',
    )

    def is_match_iata_from(self, iata):
        return is_iata_related_to_settlement_or_country(
            iata, self.settlement_from_id, self.country_from_id)

    def is_match_iata_to(self, iata):
        return is_iata_related_to_settlement_or_country(
            iata, self.settlement_to_id, self.country_to_id)


@memoize(lambda iata, settlement_id, country_id: (iata, settlement_id, country_id))
def is_iata_related_to_settlement_or_country(iata, settlement_id, country_id):
    if not settlement_id and not country_id:
        return True

    point = get_point_by_iata_or_sirena(iata)

    if not point:
        return False

    if settlement_id:
        settlement = point.get_related_settlement()
        if settlement:
            return settlement.id == settlement_id

    if country_id:
        country = point.get_related_country()
        if country:
            return country.id == country_id

    return False


@memoize(cache=CacheWithKeyTTL(30))
def _get_selfbook_rules():
    return list(SelfBookRule.objects.all().namedtuples(
        interface=SelfBookRuleInterface
    ))


@memoize(cache=CacheWithKeyTTL(30))
def _get_selfbook_rules_map_by_partner_national_version():
    return agg_with_list(
        ((partner_code, national_version), rule)
        for rule in _get_selfbook_rules()
        for partner_code in rule.get_partner_codes()
        for national_version in rule.get_national_version_codes()
    )


@memoize(
    lambda partner, national_version, iata_from, iata_to: (
        partner.code, national_version, iata_from, iata_to),
    cache=CacheWithKeyTTL(30))
def get_selfbook_company_codes(partner, national_version, iata_from, iata_to):
    allow, exclude = set(), set()

    for rule in (_get_selfbook_rules_map_by_partner_national_version()
                 .get((partner.code, national_version), [])):
        if is_rule_match_iata_direction(rule, iata_from, iata_to):
            (exclude if rule.exclude_companies else allow
             ).update(rule.get_company_codes())

    return SelfBookCompanyCodes(allow=allow, exclude=exclude)


def is_rule_match_iata_direction(rule, iata_from, iata_to):
    directions = rule.get_directions()
    if not directions:
        return True
    op = (lambda v: not v) if rule.exclude_directions else bool
    return op(any(
        (
            direction.is_match_iata_from(iata_from) and
            direction.is_match_iata_to(iata_to)
        )
        for direction in directions
    ))


SelfBookCompanyCodes = namedtuple('SelfBookCompanyCodes', ['allow', 'exclude'])


class SelfBookPartnerNationalDirectionRules(object):
    def __init__(self, partner, national_version, iata_from, iata_to):
        self._selfbook_company_codes = selfbook_company_codes = get_selfbook_company_codes(
            partner, national_version, iata_from, iata_to)

        @memoize(lambda x: x)
        def _is_selfbook_company(company_iata):
            if not (selfbook_company_codes.allow or
                    selfbook_company_codes.exclude):
                return False

            return (
                company_iata in selfbook_company_codes.allow
                or
                (selfbook_company_codes.exclude and
                 company_iata not in selfbook_company_codes.exclude)
            )

        self._is_selfbook_company = _is_selfbook_company

    def is_selfbook_applicable(self, variant):
        return all(
            self._is_selfbook_company(s.company_iata)
            for s in variant.iter_all_segments()
        )
