# -*- coding: utf-8 -*-
import os
import time
from logging import getLogger

from django.conf import settings
from flask import Flask, send_from_directory, request
from flask_swagger_ui import get_swaggerui_blueprint
from raven.contrib.flask import Sentry

from travel.avia.library.python.flask_helpers.wsgi_middleware import HTTPSchemeProxy, ABExperimentFlags
from travel.avia.ticket_daemon.ticket_daemon.settings import PROJECT_PATH
from travel.avia.ticket_daemon.ticket_daemon.views import (
    init_query_routes, redirect_routes, service_routes, track_routes, book_routes
)

log = getLogger(__name__)


def register_swagger(app):
    swagger_url = '/api/docs'
    swaggerui_blueprint = get_swaggerui_blueprint(
        swagger_url,
        '/api/docs/swagger.json',
        config={
            'app_name': 'Ticket Daemon'
        },
    )
    app.register_blueprint(swaggerui_blueprint, url_prefix=swagger_url)

    @app.route('/api/docs/swagger.json', methods=['GET'])
    def swagger():
        return send_from_directory(
            os.path.join(PROJECT_PATH, 'ticket_daemon'), 'swagger.json'
        )


class App(Flask):
    def __init__(self, *args, **kwargs):
        super(App, self).__init__(*args, **kwargs)
        self.shutdown_flag = None
        self.cache_readiness_flag = None
        self.shared_cache_readiness_flag = None


def create_app(config):

    app = App(
        __name__,
        template_folder=os.path.join(settings.PROJECT_PATH, 'ticket_daemon/templates')
    )

    app.config['JSONIFY_PRETTYPRINT_REGULAR'] = False

    if settings.SENTRY_DSN:
        app.config['SENTRY_DSN'] = settings.SENTRY_DSN
        Sentry(app)

        # Or like this
        # from raven import Client
        # from raven.middleware import Sentry
        # client = Client(settings.SENTRY_DSN)
        # application = Sentry(application, client=client)

    app.config.update(config)

    app.register_blueprint(init_query_routes)
    app.register_blueprint(redirect_routes)
    app.register_blueprint(service_routes)
    app.register_blueprint(track_routes)
    app.register_blueprint(book_routes)
    register_swagger(app)

    app.wsgi_app = HTTPSchemeProxy(app.wsgi_app)
    app.wsgi_app = ABExperimentFlags(app.wsgi_app)

    @app.route('/ping')
    def ping():
        if app.shutdown_flag.is_set():
            return 'Stopped', 410

        try:
            if not app.cache_readiness_flag.is_set():
                return 'cache_readiness_flag is not set', 503
            if not app.shared_cache_readiness_flag.is_set():
                return 'shared_cache_readiness_flag is not set', 503
        except Exception:
            log.exception('ping td exception')
            return 'error', 500

        return 'ok', 200

    @app.route('/shutdown', methods=['POST'])
    def shutdown():
        max_delay = 120  # seconds

        delay = int(request.args.get('delay', max_delay))
        if delay > max_delay:
            return 'max delay: {}'.format(max_delay), 400

        try:
            app.shutdown_flag.set()
            time.sleep(delay)
        except Exception:
            log.exception('shutdown exception')
            return 'error', 500
        else:
            return 'ok', 200

    return app
