# -*- coding: utf-8 -*-
import logging
import ujson
import zlib
from collections import namedtuple
from datetime import datetime
from itertools import chain
from typing import Dict, Set, Any, Optional

import gevent
import traceback
from django.conf import settings
from travel.avia.library.python.common.utils.iterrecipes import pairwise
from travel.avia.library.python.ticket_daemon.date import unixtime, DateTimeDeserializerV3
from travel.avia.library.python.ticket_daemon.protobuf_converting.big_wizard.search_result_converter import (
    SearchResultConverter
)
from travel.proto.avia.wizard.search_result_pb2 import SearchResult

from travel.avia.ticket_daemon.ticket_daemon.api.models_utils import popularity, get_conversion
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.partners import get_partner_by_code
from travel.avia.ticket_daemon.ticket_daemon.api.result.abstract_result import AbstractResult
from travel.avia.ticket_daemon.ticket_daemon.api.result.serializer import (
    flights_serializer, big_wizard_fares_serializer
)
from travel.avia.ticket_daemon.ticket_daemon.daemon.big_beauty_sorter import BigBeautySorter, BigBeautySortings
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_store_time_provider import partner_store_time_provider
from travel.avia.ticket_daemon.ticket_daemon.lib.saving_new_sorts_settings import saving_new_sorts_settings
from travel.avia.ticket_daemon.ticket_daemon.lib.timer import Timeline, TimelineWithDefaults
from travel.avia.ticket_daemon.ticket_daemon.lib.ydb.wizard_cache import WizardCache
from travel.avia.ticket_daemon.ticket_daemon.lib.ydb.wizard_cache_by_partner import WizardCacheByPartner
from travel.avia.ticket_daemon.ticket_daemon.lib.ydb.wizard_cache_experimental import WizardCacheExperimental

log = logging.getLogger(__name__)

MAX_PRICE = 2 ** 31 - 1


def _model_to_serialize(model):
    return model and model.id


def _with_baggage(baggage):
    """

    :param baggage: Example - [['1d1d23d'], ['1p1p15p', '1p1pN']]
    :rtype: bool
    """
    return all(b and b.startswith('1') for b in chain.from_iterable(baggage))


def _one_defined_airline(fare_airlines):
    return len(fare_airlines) == 1 and None not in fare_airlines


def gevent_joinall_functions(*funcs):
    gevent.joinall([
        gevent.spawn(func) for func in funcs
    ])


def get_stored_search_result(query, is_experimental=False):
    search_result = SearchResult()

    try:
        log.info('Read big beauty variants from ydb: %s', query.key())

        if is_experimental:
            wizard_result = WizardCacheExperimental().get(query)
        else:
            wizard_result = WizardCache().get(query)
        if wizard_result:
            search_result.ParseFromString(zlib.decompress(wizard_result[0]['search_result']))
    except Exception:
        log.exception('Exception on getting stored big beauty results')

    return search_result


def get_stored_search_results_by_partner(query, partner_codes):
    search_results = {code: SearchResult() for code in partner_codes}

    try:
        log.info('Read big beauty variants by partner from ydb: %s by %s', query.key(), ','.join(partner_codes))

        result_by_partners = WizardCacheByPartner().get_all(query, partner_codes)
        for result_by_partner in result_by_partners:
            search_results[result_by_partner['partner_code']].ParseFromString(zlib.decompress(result_by_partner['search_result']))

    except Exception as e:
        log.exception('Exception on getting stored big beauty results by partner %r', e)

    return search_results


class BigBeautyCollector(object):
    """
    Хранит все варианты с минимальными ценами по запросу
    Инициализируется 1 на сессию опроса партнеров и накапливает в себе ответы по всем партнерам
    """

    def __init__(self, query, saved_search_result, search_result_logger, is_experimental=False):
        """

        :type query: ticket_daemon.api.query.Query
        :type saved_search_result: SearchResult
        :type search_result_logger: travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.wizard_search_result_logger.WizardSearchResultLogger
        """
        self.timeline_logger = TimelineWithDefaults(
            Timeline(log), defaults={'qid': query.id}
        )
        self.search_result_logger = search_result_logger
        self._query = query
        stored = SearchResultConverter().to_dictionary(saved_search_result)

        self._version = stored['version']
        self._flights = stored['flights']
        self._variants_by_route = self._init_stored_fares(query, stored)
        self._polling_status = PollingStatus(query.partners)
        self._updates = 0
        self._is_experimental = is_experimental
        self.partners = set()

    @property
    def is_experimental(self):
        return self._is_experimental

    def add(self, result):
        if not getattr(result.partner, 'enabled_in_wizard_%s' % self._query.national_version, False):
            return

        variant_updates = self._add_variants(result)
        if variant_updates:
            self._updates += variant_updates

        if variant_updates or self._add_partner_for_filter(result):
            self.store()  # Тут можно обновлять, когда апдэйтов пришло ощутимое количество, а не на каждый пук дёргать саас

    def _add_partner_for_filter(self, result):
        if result.variants:
            updated = result.partner.code not in self.partners
            self.partners.add(result.partner.code)
            return updated

    def _add_variants(self, result):
        """
        :type result: ticket_daemon.api.result.result.Result
        """
        if not self._need_add_variants_by_partner(result.partner.code):
            return

        variants = big_wizard_fares_serializer.serialize(
            result.variants, result.created, result.expire
        )
        flights = flights_serializer.serialize(result.variants)
        updates = 0
        now = unixtime()
        cache_ttl = partner_store_time_provider.get_result_time(result.partner, None)
        partner_conversion = self._get_conversion(result.partner.code)

        for variant in variants:
            routes = variant['route']
            baggage_tariff = self._get_baggage_tariff(variant['baggage'])
            tariff_info = big_wizard_fares_serializer.tariff_info(variant)

            tariff_info['conversion_partner'] = tariff_info['partner']
            variant['conversion_partner'] = variant['partner']

            if routes not in self._variants_by_route:
                self._variants_by_route[routes] = variant
                self._variants_by_route[routes]['tariffs'][baggage_tariff] = tariff_info
                updates += 1
                for route in chain.from_iterable(routes):
                    self._flights[route] = flights[route]
            else:
                saved_variant = self._variants_by_route[routes]
                saved_tariff = saved_variant['tariff']['value']
                saved_expire = saved_variant['created'] + cache_ttl
                saved_baggage_tariffs = saved_variant['tariffs']
                saved_same_baggage_tariff = saved_baggage_tariffs.get(baggage_tariff)

                tariff = variant['tariff']['value']
                if (
                    tariff < saved_tariff
                    or (variant['partner'] == saved_variant['partner'] and saved_expire < now)
                    or (tariff == saved_tariff and result.partner.is_aviacompany)
                ):
                    self._variants_by_route[routes] = variant
                    self._variants_by_route[routes]['tariffs'] = saved_baggage_tariffs
                    self._variants_by_route[routes]['tariffs'][baggage_tariff] = tariff_info
                    updates += 1

                elif (
                    not saved_same_baggage_tariff
                    or saved_same_baggage_tariff
                    and (
                        tariff < saved_same_baggage_tariff['price']['value']
                        or (
                            variant['partner'] == saved_same_baggage_tariff['partner']
                            and saved_same_baggage_tariff['created_at'] + cache_ttl < now
                        )
                        or (
                            tariff == saved_same_baggage_tariff['price']['value']
                            and result.partner.is_aviacompany
                        )
                    )
                ):
                    self._variants_by_route[routes]['tariffs'][baggage_tariff] = tariff_info
                    updates += 1

                if (
                    tariff == saved_tariff
                    and not result.partner.is_aviacompany
                    and not self._is_aviacompany_partner(saved_variant['partner'])
                    and partner_conversion > self._get_conversion(saved_variant['conversion_partner'])
                ):
                    self._variants_by_route[routes]['conversion_partner'] = result.partner.code
                    self._variants_by_route[routes]['tariffs'][baggage_tariff]['conversion_partner'] = result.partner.code
                    updates += 1
                elif (
                    saved_same_baggage_tariff
                    and tariff == saved_same_baggage_tariff['price']['value']
                    and not result.partner.is_aviacompany
                    and not self._is_aviacompany_partner(saved_same_baggage_tariff['partner'])
                    and partner_conversion > self._get_conversion(saved_same_baggage_tariff.get('conversion_partner'))
                ):
                    self._variants_by_route[routes]['tariffs'][baggage_tariff]['conversion_partner'] = result.partner.code
                    updates += 1

        self.timeline_logger.event('updates', extra={'count': updates, 'partner': result.partner.code})
        return updates

    def store(self):
        self._version += 1
        result = BigBeautyResult(
            query=self._query,
            flights=self._flights,
            fares=self._variants_by_route,
            version=self._version,
            polling_status=self._polling_status,
            partners=self.partners,
            is_experimental=self._is_experimental,
        )
        result.store()

    @staticmethod
    def _get_baggage_tariff(baggage):
        """

        :return: 'with_baggage' 'without_baggage'
        """
        return 'with_baggage' if _with_baggage(baggage) else 'without_baggage'

    @staticmethod
    def _get_conversion(partner_code):
        partner = get_partner_by_code(partner_code)
        billing_order_id = getattr(partner, 'billing_order_id', None)
        return get_conversion(billing_order_id)

    @staticmethod
    def _is_aviacompany_partner(partner_code):
        partner = get_partner_by_code(partner_code)
        return getattr(partner, 'is_aviacompany', False)

    def _init_stored_fares(self, query, stored):
        """
        Удалить все сохраненные варианты по партнерам, которые
         переопрашиваются в текущей сессии поиска и те, у которых прошло время жизни.
        """
        partners = {p.code for p in query.partners}
        unix_time = unixtime()
        variants_by_route = {}

        idx = 0
        for fare in stored['fares']:
            idx += 1
            actual_tariffs = {
                tariff_name: tariff for tariff_name, tariff in fare['tariffs'].iteritems()
                if tariff['partner'] not in partners or tariff['expire_at'] > unix_time
            }
            if len(actual_tariffs) == 0:
                continue
            elif len(actual_tariffs) < fare['tariffs']:
                min_tariff = min(actual_tariffs.itervalues(), key=lambda variant: variant['price']['value'])
                fare.update({
                    'tariff': min_tariff['price'],
                    'partner': min_tariff['partner'],
                    'baggage': min_tariff['baggage'],
                    'created': min_tariff['created_at'],
                    'expire': min_tariff['expire_at'],
                    'tariffs': actual_tariffs
                })

            variants_by_route[fare['route']] = fare

        self.timeline_logger.event(
            'init stored routes',
            extra={'left': len(variants_by_route), 'all': idx, 'version': stored['version']}
        )
        return variants_by_route

    def partner_done(self, partner):
        self._polling_status.finalize(partner.code)
        if self.all_replied:
            self.timeline_logger.event(
                'All partners done',
                extra={'updates': self._updates, 'all': len(self._variants_by_route), 'version': self._version}
            )

            self.search_result_logger.log(
                query=self._query,
                variants_count=len(self._variants_by_route),
                min_price=self._get_min_price(),
                test_id=self._query.meta.get('test_id') or '',
                is_experimental=self._is_experimental,
            )

            self.store()

    @property
    def all_replied(self):
        """Все ли ответили"""
        return bool(
            not self._polling_status.remaining_partners
        )

    def _need_add_variants_by_partner(self, partner_code):
        return partner_code not in settings.DISABLED_PARTNERS_FOR_WIZARD_COMMON_CACHE

    def _get_min_price(self):
        if self._variants_by_route:
            return min(v['tariff']['value'] for v in self._variants_by_route.itervalues())
        return None


class BigBeautyCollectorByPartner(BigBeautyCollector):
    def __init__(self, query, saved_search_result, search_result_logger, partner_code):
        super(BigBeautyCollectorByPartner, self).__init__(query, saved_search_result, search_result_logger)
        self.partner_code = partner_code

    def store(self):
        try:
            self._version += 1
            if self._variants_by_route:
                result = BigBeautyResultByPartner(
                    query=self._query,
                    flights=self._flights,
                    fares=self._variants_by_route,
                    version=self._version,
                    polling_status=self._polling_status,
                    partners=self.partners,
                    partner_code=self.partner_code,
                )
                result.store()
        except Exception as exc:
            log.exception(u'Exception %r on BigBeautyCollectorByPartner.store', exc)

    def partner_done(self, partner):
        self._polling_status.finalize(partner.code)
        if not self._need_add_variants_by_partner(partner.code):
            return
        try:
            self.store()
        except Exception as exc:
            log.exception(u'Exception %r on BigBeautyCollectorByPartner.partner_done', exc)

    def _need_add_variants_by_partner(self, partner_code):
        return partner_code == self.partner_code


class BigBeautyResult(AbstractResult):
    __columns = ('filter_state', 'protobuf')
    COLUMNS = namedtuple('COLUMNS', __columns)(*__columns)

    STORE_TIME = 15 * 60 * 60  # 15 hours
    EXPERIMENTAL_STORE_TIME = 3 * 24 * 60 * 60  # 3 days
    converter = SearchResultConverter()
    wizard_cache = WizardCache()
    wizard_cache_experimental = WizardCacheExperimental()

    def cache_key(self, *args, **kwargs):
        raise NotImplementedError

    def __init__(self, query, flights, fares, version, polling_status, partners, is_experimental=False):
        """

        :param dict flights: {flight_tag: flight}
        :param dict fares: {route: fare}
        :type polling_status: PollingStatus
        :param int version:
        """
        self._query = query
        self._flights = flights
        self._fares = self._sort_and_set_popularities(fares.itervalues(), flights)
        self._version = version
        self._polling_status = polling_status
        self._is_experimental = is_experimental
        self.sorter = BigBeautySorter.create()
        self._partners = partners

    def to_dict(self):
        return {
            'qid': self._query.id,
            'flights': self._flights,
            'fares': self._fares,
            'version': self._version,
            'offers_count': len(self._fares),
            'polling_status': self._polling_status.to_dict(),
        }

    @staticmethod
    def _to_protobuf_safe(search_result_dict):
        # type: (Any) -> Optional[Any]

        try:
            return BigBeautyResult.converter.to_protobuf(search_result_dict).SerializeToString()
        except Exception:
            log.error(
                'Failed to convert big_beauty_result. %s',
                traceback.format_exc(),
            )

        return None

    @staticmethod
    def _compress(proto_string):
        # type: (Optional[str]) -> Optional[str]
        if not proto_string or not isinstance(proto_string, str):
            return proto_string
        try:
            return zlib.compress(proto_string)
        except Exception:
            log.error(
                'Failed to compress wizard search_result. %s',
                traceback.format_exc(),
            )
        return proto_string

    @staticmethod
    def _get_custom_sorting_safe(search_result, sort_name, sorter, query):
        try:
            sorted_search_result = sorter.sort(search_result, query, sort_name)
            return BigBeautyResult._to_protobuf_safe(sorted_search_result)
        except Exception:
            log.error(
                'Failed to apply custom sort %s. %s',
                sort_name,
                traceback.format_exc(),
            )

    def pack_search_results(self):
        default_search_result = self.to_dict()
        new_sorts_settings = saving_new_sorts_settings()

        result = {}
        result[BigBeautySortings.PROTOBUF] = BigBeautyResult._compress(BigBeautyResult._to_protobuf_safe(default_search_result))

        if new_sorts_settings.save_sorted_by_price:
            result[BigBeautySortings.SORTED_BY_PRICE] = BigBeautyResult._get_custom_sorting_safe(
                default_search_result,
                BigBeautySortings.SORTED_BY_PRICE,
                self.sorter,
                self._query
            )

        if new_sorts_settings.save_control_with_weekdays:
            result[BigBeautySortings.CONTROL_WITH_WEEKDAYS] = BigBeautyResult._get_custom_sorting_safe(
                default_search_result,
                BigBeautySortings.CONTROL_WITH_WEEKDAYS,
                self.sorter,
                self._query
            )

        if new_sorts_settings.save_front_sort:
            result[BigBeautySortings.FRONT_SORT] = BigBeautyResult._get_custom_sorting_safe(
                default_search_result,
                BigBeautySortings.FRONT_SORT,
                self.sorter,
                self._query
            )

        if new_sorts_settings.save_kateov_sort:
            result[BigBeautySortings.KATEOV_SORT] = BigBeautyResult._get_custom_sorting_safe(
                default_search_result,
                BigBeautySortings.KATEOV_SORT,
                self.sorter,
                self._query
            )

        return result

    def create_result(self):
        search_results = self.pack_search_results()
        search_results[BigBeautyResult.COLUMNS.filter_state] = FilterState(flights=self._flights, fares=self._fares, partners=self._partners).pack()
        return search_results

    def store(self):
        search_results = self.create_result()

        if not self._is_experimental:
            self._store_into_ydb(search_results, self.STORE_TIME)
        else:
            self._store_into_ydb_experimental(search_results, self.EXPERIMENTAL_STORE_TIME)

    def _store_into_ydb(self, search_results, store_time):
        try:
            search_result = search_results[BigBeautySortings.PROTOBUF]
            filter_state = search_results[BigBeautyResult.COLUMNS.filter_state]
            min_price = self.get_min_price(self._fares)
            expires_at_by_partner = self.get_expires_at_by_partner(self._fares)
            expires_at = max([expires_at_by_partner, unixtime() + store_time])
            ttl_expires_at = datetime.fromtimestamp(expires_at)

            log.info(
                'Saving to ydb %s: min_price=%s, expired_at=%s, expires_at_by_partner=%s, ttl_expires_at=%s',
                self._query.key(), min_price, expires_at, expires_at_by_partner, ttl_expires_at,
            )

            self.wizard_cache.set(
                self._query,
                search_result,
                filter_state,
                min_price,
                expires_at,
                expires_at_by_partner,
                ttl_expires_at,
            )
        except Exception:
            log.error(
                'Failed to store into ydb. %s',
                traceback.format_exc(),
            )

    def _store_into_ydb_experimental(self, search_results, store_time):
        try:
            search_result = search_results[BigBeautySortings.PROTOBUF]
            filter_state = search_results[BigBeautyResult.COLUMNS.filter_state]
            min_price = self.get_min_price(self._fares)
            expires_at_by_partner = self.get_expires_at_by_partner(self._fares)
            expires_at = max([expires_at_by_partner, unixtime() + store_time])
            ttl_expires_at = datetime.fromtimestamp(expires_at)
            created_at = unixtime()

            log.info(
                'Saving to ydb %s: min_price=%s, expired_at=%s, created_at=%s, ttl_expires_at=%s',
                self._query.key(), min_price, expires_at, created_at, ttl_expires_at,
            )

            self.wizard_cache_experimental.set(
                self._query,
                search_result,
                filter_state,
                min_price,
                expires_at,
                created_at,
                ttl_expires_at,
            )
        except Exception:
            log.error(
                'Failed to store into ydb. %s',
                traceback.format_exc(),
            )

    @staticmethod
    def get_expires_at_by_partner(fares):
        if fares:
            return max(v['expire'] for v in fares)
        return 0

    @staticmethod
    def get_min_price(fares):
        if fares:
            return int(min(v['tariff']['value'] for v in fares) * 100)
        return MAX_PRICE

    @staticmethod
    def _sort_and_set_popularities(variants, flights):
        """
        Отбираем:
        1. Самые дешевый вариант;
        2. Самые популярные из оставшихся.
        При совпадении ключа сравнения предпочитаем самые дешёвые варианты.
        """

        def _get_popularity(variant):
            return sum(
                popularity(flights[route]['number'] for route in routes)
                for routes in variant['route']
            )

        def _get_variant_with_popularity(variant):
            variant['popularity'] = _get_popularity(variant)
            return variant

        cheapest = sorted(variants, key=lambda x: x['tariff']['value'])
        if not cheapest:
            return []
        cheapest = (_get_variant_with_popularity(variant) for variant in cheapest)

        # Пользуемся стабильностью сортировки
        return [next(cheapest)] + sorted(cheapest, key=lambda x: x['popularity'], reverse=True)

    @staticmethod
    def make_key(qkey, lang):
        return '{}{}/{}_big_beauty'.format(
            settings.TICKET_DAEMON_CACHEROOT, qkey, lang
        )


class BigBeautyResultByPartner(BigBeautyResult):
    wizard_cache_by_partner = WizardCacheByPartner()

    def cache_key(self, *args, **kwargs):
        super(BigBeautyResultByPartner, self).cache_key()

    def __init__(self, query, flights, fares, version, polling_status, partners, partner_code):
        super(BigBeautyResultByPartner, self).__init__(query, flights, fares, version, polling_status, partners)
        self.partner_code = partner_code

    def store(self):
        search_results = self.create_result()
        self._store_into_ydb(search_results, self.STORE_TIME)

    def _store_into_ydb(self, search_results, store_time):
        try:
            search_result = search_results[BigBeautySortings.PROTOBUF]
            filter_state = search_results[BigBeautyResult.COLUMNS.filter_state]
            min_price = self.get_min_price(self._fares)
            expires_at_by_partner = self.get_expires_at_by_partner(self._fares)
            expires_at = max([expires_at_by_partner, unixtime() + store_time])
            ttl_expires_at = datetime.fromtimestamp(expires_at)

            log.info(
                'Saving to ydb by partner %s: min_price=%s, expired_at=%s, expires_at_by_partner=%s, ttl_expires_at=%s',
                self._query.key(), min_price, expires_at, expires_at_by_partner, ttl_expires_at,
            )

            worker = gevent.spawn(
                self.wizard_cache_by_partner.set,
                self._query,
                search_result,
                filter_state,
                min_price,
                expires_at,
                expires_at_by_partner,
                ttl_expires_at,
                self.partner_code,
            )
            gevent.wait([worker])
        except Exception:
            log.error(
                'Failed to store by partner into ydb. %s',
                traceback.format_exc(),
            )


class FilterState(object):
    def __init__(self, result=None, flights=None, fares=None, partners=None):
        """
        :type result: BigBeautyResult

        """
        if result:
            self._flights = result._flights
            self._fares = result._fares
        else:
            self._flights = flights
            self._fares = fares

        self._partners = partners if partners is not None else set()
        self._dt_deserializer = DateTimeDeserializerV3()

    def pack(self):
        return ujson.dumps(self.to_dict())

    def to_dict(self):
        result = self._default_result()

        try:
            self._update_result(result)
            result['partners'] = self._partners
        except Exception:
            log.warning('Failed to fill bb_filters%s', traceback.format_exc())
        return result

    def _update_result(self, result):
        for fare in self._fares:
            forward, backward = fare['route']
            self._fill(result, forward, 'forward')
            if backward:
                self._fill(result, backward, 'backward')

            fare_airlines = {
                self._flights.get(route, {}).get('company')
                for route in chain.from_iterable(fare['route'])
            }
            if _one_defined_airline(fare_airlines):
                result['airlines'].update(fare_airlines)

            self._update_minprices(result, fare, fare_airlines)

        for flight in self._flights.itervalues():
            if flight['company']:
                result['all_airlines'].add(flight['company'])

    @staticmethod
    def _default_result():
        return {
            'withBaggage': None,
            'prices': {
                'withBaggage': None,
                'directFlight': None,
                'airports': {'from': dict(), 'to': dict()},
                'airlines': dict(),
                'directAirlines': set(),
                'transfers': {0: None, 1: None, 2: None},
            },
            'transfer': {
                'count': None,
                'minDuration': None,
                'maxDuration': None,
                'hasAirportChange': None,
                'hasNight': None,
            },
            'time': {
                # Время отправления в прямом направлении: range
                'forwardDepartureMin': None,
                'forwardDepartureMax': None,
                # Время прибытия в прямом направлении: range
                'forwardArrivalMin': None,
                'forwardArrivalMax': None,
                # Время отправления в обратном направлении: range
                'backwardDepartureMin': None,
                'backwardDepartureMax': None,
                # Время прибытия в обратном направлении: range
                'backwardArrivalMin': None,
                'backwardArrivalMax': None,
            },
            'airport': {
                # Аэропорты отправления в прямом направлении: список
                'forwardDeparture': set(),
                # Аэропорты прибытия в прямом направлении: список
                'forwardArrival': set(),
                # Аэропорты пересадок на перелет в прямом направлении: список
                'forwardTransfers': set(),
                # Аэропорты отправления в обратном направлении: список
                'backwardDeparture': set(),
                # Аэропорты прибытия в обратном направлении: список
                'backwardArrival': set(),
                # Аэропорты пересадок на перелет в обратном направлении: список
                'backwardTransfers': set(),
            },
            'airlines': set(),  # Список авиакомпаний для фильтра
            'all_airlines': set(),  # Список всех авиакомпаний предложений
            'partners': set(),  # Список партнеров для фильтра
        }

    def _fill(self, result, routes, direction):
        flights = map(self._flights.__getitem__, chain(routes))

        departure_key = '%sDeparture' % direction
        arrival_key = '%sArrival' % direction

        result['airport'][departure_key].add(flights[0]['from'])
        result['airport'][arrival_key].add(flights[-1]['to'])
        try:
            self.update_time(result, departure_key, flights[0]['departure']['local'])
        except KeyError:
            log.warning('Can not update %s for flight %r', departure_key, flights[0])

        try:
            self.update_time(result, arrival_key, flights[-1]['arrival']['local'])
        except KeyError:
            log.warning('Can not update %s for flight %r', arrival_key, flights[-1])

        update_by_pred(result['transfer'], 'count', len(flights) - 1, max)

        if len(flights) > 1:
            for f1, f2 in pairwise(flights):
                result['airport']['%sTransfers' % direction].add(f1['to'])
                result['airport']['%sTransfers' % direction].add(f2['from'])

                # обновляем всё что нужно по времени пересадок (transfer)
                try:
                    transfer_duration = (
                        self._dt_deserializer.deserialize(f2['departure']['local'])
                        - self._dt_deserializer.deserialize(f1['arrival']['local'])
                    ).total_seconds() / 60
                except KeyError:
                    log.warning('Can not calculate transfer duration for flights %r, %r', f1, f2)
                    continue

                if transfer_duration < 0:
                    log.warning('Negative transfer duration %d for flights %r, %r', transfer_duration, f1, f2)
                    continue
                update_by_pred(result['transfer'], 'minDuration', transfer_duration, min)
                update_by_pred(result['transfer'], 'maxDuration', transfer_duration, max)

    @staticmethod
    def update_time(result, key, timestamp):
        min_key = key + 'Min'
        max_key = key + 'Max'
        update_by_pred(result['time'], min_key, timestamp, min)
        update_by_pred(result['time'], max_key, timestamp, max)

    @classmethod
    def _min_tariff(cls, first, second):
        if not first and not second:
            return None

        if not first:
            return second

        if not second:
            return first

        return min(first, second, lambda x: x['value'])

    def _update_minprices(self, result, fare, fare_airlines):
        # type: (Dict[str, Any], Dict[str, Any], Set[int]) -> None

        forward, backward = fare['route']

        is_direct_flight = len(forward) == 1 and len(backward) <= 1
        if is_direct_flight:
            result['prices']['directFlight'] = self._min_tariff(result['prices']['directFlight'], fare['tariff'])

        if _with_baggage(fare['baggage']):
            result['prices']['withBaggage'] = self._min_tariff(result['prices']['withBaggage'], fare['tariff'])

        if _one_defined_airline(fare_airlines):
            airline = next(iter(fare_airlines))
            result['prices']['airlines'][airline] = self._min_tariff(
                result['prices']['airlines'].get(airline),
                fare['tariff']
            )
            if is_direct_flight:
                result['prices']['directAirlines'].add(airline)

        forward_flights = map(self._flights.__getitem__, forward)
        backward_flights = map(self._flights.__getitem__, backward)

        max_transfers_count = max(len(forward_flights) - 1, len(backward_flights) - 1 if backward_flights else 0)

        for transfers_count in range(max_transfers_count, 3):
            result['prices']['transfers'][transfers_count] = self._min_tariff(
                result['prices']['transfers'][transfers_count],
                fare['tariff']
            )

        if not backward_flights or (
            backward_flights and
            forward_flights[0]['from'] == backward_flights[-1]['to']
        ):
            from_airport = forward_flights[0]['from']
            result['prices']['airports']['from'][from_airport] = self._min_tariff(
                result['prices']['airports']['from'].get(from_airport),
                fare['tariff']
            )

        if not backward_flights or (
            backward_flights and
            forward_flights[-1]['to'] == backward_flights[0]['from']
        ):
            to_airport = forward_flights[-1]['to']
            result['prices']['airports']['to'][to_airport] = self._min_tariff(
                result['prices']['airports']['to'].get(to_airport),
                fare['tariff']
            )


def update_by_pred(d, key, value, pred):
    if d[key] is None:
        d[key] = value
    else:
        d[key] = pred(value, d[key])


class PollingStatus(object):
    def __init__(self, partners):
        self.asked_partners = set()
        self.remaining_partners = {p.code for p in partners}

    def to_dict(self):
        return {
            'asked_partners': list(sorted(self.asked_partners)),
            'asked_partners_count': len(self.asked_partners),
            'remaining_partners': list(sorted(self.remaining_partners)),
            'remaining_partners_count': len(self.remaining_partners),
        }

    def finalize(self, p_code):
        self.asked_partners.add(p_code)
        if p_code in self.remaining_partners:
            self.remaining_partners.remove(p_code)
