# -*- coding: utf-8 -*-
import logging
from copy import copy
from itertools import chain, islice
from datetime import datetime
from typing import Any, Dict, List, Iterable

from travel.avia.ticket_daemon.ticket_daemon.api.models_utils import popularity_by_weekday_and_national_version
from travel.avia.ticket_daemon.ticket_daemon.api.partners_utils import get_partner_by_code
from travel.avia.ticket_daemon.ticket_daemon.api.query import Query
from travel.avia.ticket_daemon.ticket_daemon.daemon.extended_fares.extended_fares_comparator import ExtendedFaresComparator, ExtendedFaresComparingResult
from travel.avia.ticket_daemon.ticket_daemon.daemon.extended_fares.fare_extender import FareExtender

log = logging.getLogger(__name__)


class BigBeautySortings(object):
    PROTOBUF = 'protobuf'
    SORTED_BY_PRICE = 'sorted_by_price'
    CONTROL_WITH_WEEKDAYS = 'control_with_weekdays'
    FRONT_SORT = 'front_sort'
    KATEOV_SORT = 'kateov_sort'


class BigBeautySorter(object):
    def __init__(self, fare_extender, extended_fares_comparator):
        # type: (FareExtender, ExtendedFaresComparator) -> None

        self._fare_extender = fare_extender
        self._extended_fares_comparator = extended_fares_comparator

    @staticmethod
    def create():
        return BigBeautySorter(FareExtender(), ExtendedFaresComparator())

    def _set_weekday_popularity_and_sort(self, search_result, query):
        # type: (Any, Query) -> None

        def _get_popularity(fare):
            forward_popularity = (
                popularity_by_weekday_and_national_version(
                    (search_result['flights'][route]['number'] for route in fare['route'][0]),
                    query.date_forward.weekday(),
                    query.national_version
                )
            )

            backward_popularity = 0
            if query.date_backward:
                backward_popularity = (
                    popularity_by_weekday_and_national_version(
                        (search_result['flights'][route]['number'] for route in fare['route'][1]),
                        query.date_backward.weekday(),
                        query.national_version
                    )
                )

            return forward_popularity + backward_popularity

        for f in search_result['fares']:
            f['popularity'] = _get_popularity(f)

        search_result['fares'].sort(key=lambda x: -x['popularity'])
        search_result['fares'] = self._place_cheapest_fare_on_first_position(search_result['fares'])

    def _search_result_with_new_fares(self, search_result, new_fares):
        return {
            'qid': search_result['qid'],
            'flights': search_result['flights'],
            'fares': new_fares,
            'version': search_result['version'],
            'offers_count': search_result['offers_count'],
            'polling_status': search_result['polling_status'],
        }

    def _search_result_with_copied_fares(self, search_result):
        return self._search_result_with_new_fares(search_result, copy(search_result['fares']))

    def control_with_weekdays(self, search_result, query):
        """
        Sorts fares by popularity just like now but considering weekdays and national version.
        """
        search_result = self._search_result_with_copied_fares(search_result)

        self._set_weekday_popularity_and_sort(search_result, query)

        return search_result

    def _fare_is_selected(self, selected_fares, new_fare):
        for f in selected_fares:
            if f is new_fare:
                return True

        return False

    def _first_non_selected(self, fares, selected_fares, predicate=None):
        for f in fares:
            if predicate and not predicate(f):
                continue

            if not self._fare_is_selected(selected_fares, f):
                return f

        return None

    def _transfers_count(self, fare):
        # type: (Any) -> int

        transfers_count = len(fare['route'][0]) - 1

        # Check whether backward segments exist
        if fare['route'][1]:
            transfers_count += len(fare['route'][1]) - 1

        return transfers_count

    def _direction_has_airport_change(self, route, flights):
        # type: (Any) -> bool

        for i in range(len(route) - 1):
            first_flight = flights[route[i]]
            second_flight = flights[route[i + 1]]

            if first_flight['to'] != second_flight['from']:
                return True

        return False

    def _has_airport_change(self, fare, flights):
        # type: (Any, Any) -> bool

        return (
            self._direction_has_airport_change(fare['route'][0], flights) or
            self._direction_has_airport_change(fare['route'][1], flights)
        )

    def _time_difference(self, first_time_point, second_time_point):
        # type: (Any, Any) -> float

        first_datetime = datetime.strptime(first_time_point['local'], '%Y-%m-%dT%H:%M:%S')
        second_datetime = datetime.strptime(second_time_point['local'], '%Y-%m-%dT%H:%M:%S')

        datetime_absolute_diff = abs((first_datetime - second_datetime).total_seconds() / 60)
        offset_absolute_diff = abs(first_time_point['offset'] - second_time_point['offset'])

        return datetime_absolute_diff - offset_absolute_diff

    def _fare_duration(self, fare, flights):
        # type: (Any) -> float

        duration = self._time_difference(
            flights[fare['route'][0][0]]['departure'],
            flights[fare['route'][0][-1]]['arrival'],
        )

        if fare['route'][1]:
            duration += self._time_difference(
                flights[fare['route'][1][0]]['departure'],
                flights[fare['route'][1][-1]]['arrival'],
            )

        return duration

    def _get_comfortable_fare(self, fares, flights):
        # type: (List[Any], Any) -> Optional[Any]

        # Here we have to choose the most comfortable fares.
        # Algorithm for determining the most comfortable fares is the following:
        # 1. Remove all fares which transfers count exceeds minimal transfers count by 2.
        # 2. Remove all fares with airports changes if it does not remove all the variants.
        # 3. Remove all fares which duration exceeds fastest fare duration (among remaining fares) by DELTA.
        #    Here DELTA is the maximum of 15 minutes and 20% of fastest remaining fare duration.
        # 4. Sort remaining fares by duration in ascending order.
        # 5. Choose cheapest fare among remaining fares.

        if not fares:
            return None

        # Step 1
        minimum_transfers_count = min(self._transfers_count(f) for f in fares)
        fares = [f for f in fares if self._transfers_count(f) < minimum_transfers_count + 2]

        # Step 2
        fares_without_airport_changes = [f for f in fares if not self._has_airport_change(f, flights)]
        if fares_without_airport_changes:
            fares = fares_without_airport_changes

        # Step 3
        minimum_duration = min(self._fare_duration(f, flights) for f in fares)
        delta = max(0.2 * minimum_duration, 15)
        allowed_duration = minimum_duration + delta

        fares = [f for f in fares if self._fare_duration(f, flights) < allowed_duration]

        # Step 4, 5
        return min(fares, key=lambda f: (f['tariff']['value'], self._fare_duration(f, flights)))

    def _from_aviacompany(self, fare):
        if not fare:
            return False

        partner = get_partner_by_code(fare['partner'])
        if not partner:
            return False

        return partner.is_aviacompany

    def _with_baggage(self, fare):
        # type: (Any) -> bool

        return all(b and b.startswith('1') for b in chain.from_iterable(fare['baggage']))

    def _reorder_fares(self, fares, prior_fares):
        reordered_fares = copy(prior_fares)

        for f in fares:
            if not self._fare_is_selected(prior_fares, f):
                reordered_fares.append(f)

        return reordered_fares

    def _cheapest_position(self, fares):
        # type: (List[Any]) -> int

        if not fares:
            return -1

        position, fare = min(enumerate(fares), key=lambda (index, fare): fare['tariff']['value'])
        return position

    def _place_cheapest_fare_on_first_position(self, fares):
        # type: (List[Any]) -> List[Any]

        if not fares:
            return fares

        cheapest_position = self._cheapest_position(fares)

        if cheapest_position == 0:
            return fares

        return list(chain([fares[cheapest_position]], islice(fares, cheapest_position), islice(fares, cheapest_position + 1, None)))

    def front_sort(self, search_result, query):
        """
        Here we should return fares in order like on our website avia.yandex.net.

        Variants should be reorder in the following manner:
        1. Cheapest fare comes first
        2. The most 'comfortable' fare among remaining fares
        3. The most popular fare among remaining fares. Here popularity of flights depends on national_version and weekday.
        4. If there is no fare from aviacompany in selected fares add most popular fare among remaining fares from aviacompanies.
        5. If there is no fare with baggage in selected fares add most popular fare among remaining fares with baggage.
        """

        search_result = self._search_result_with_copied_fares(search_result)
        self._set_weekday_popularity_and_sort(search_result, query)

        fares = search_result['fares']

        if not fares:
            return search_result

        # The cheapest fare
        selected_fares = [min(fares, key=lambda f: f['tariff']['value'])]

        # The most comfortable among remaining fares
        next_fare = self._first_non_selected(
            [self._get_comfortable_fare(fares, search_result['flights'])],
            selected_fares
        )
        if next_fare:
            selected_fares.append(next_fare)

        # The most popular among remaining fares
        next_fare = self._first_non_selected(fares, selected_fares)
        if next_fare:
            selected_fares.append(next_fare)

        # Fare from aviacompany if did not select yet
        if not any(self._from_aviacompany(f) for f in selected_fares):
            next_fare = self._first_non_selected(fares, selected_fares, predicate=self._from_aviacompany)
            if next_fare:
                selected_fares.append(next_fare)

        # Fare with baggage if did not select yet
        if not any(self._with_baggage(f) for f in selected_fares):
            next_fare = self._first_non_selected(fares, selected_fares, predicate=self._with_baggage)
            if next_fare:
                selected_fares.append(next_fare)

        result = self._search_result_with_new_fares(search_result, self._reorder_fares(fares, selected_fares))

        return result

    def sort_by_price(self, search_result):
        """
        Sorts fares simply by price.
        """

        sorted_fares = sorted(search_result['fares'], key=lambda x: x['tariff']['value'])

        return {
            'qid': search_result['qid'],
            'flights': search_result['flights'],
            'fares': sorted_fares,
            'version': search_result['version'],
            'offers_count': search_result['offers_count'],
            'polling_status': search_result['polling_status'],
        }

    def _extend_fares(self, fares, flights):
        # type: (Iterable[Any], Dict[str, Any]) -> Iterable[Any]

        for fare in fares:
            yield self._fare_extender.extend(fare, flights)

    def _update_good_fares(self, good_fares, next_fare_index, next_fare):
        # type: (Dict[int, Any], int, Any) -> None

        # Method receives dictionary of good fares and a good fare candidate.
        # Here we have to decide whether it is necessary to add next fare and to remove some of existing fares.
        # Method returns None as it changes incoming dictionary.

        # Method algorithm is the following:
        # If some fare among good fares is better than candidate fare then nothing changes.
        # If candidate fare is better than any of good fares or candidate fare is as good as any good fare (loog BOTH)
        #   then we have to add candidate fare and remove some of good fares.
        # To be precise we have to remove all fares from good which are worse than candidate or can be substituted by candidate (look ANY).

        good_fares_indices_to_remove = []

        necessary_to_insert_next_fare = False

        for good_fare_index, good_fare in good_fares.iteritems():
            compare_result = self._extended_fares_comparator.compare(good_fare, next_fare)

            if compare_result == ExtendedFaresComparingResult.FIRST:
                return

            if (
                compare_result == ExtendedFaresComparingResult.SECOND or
                compare_result == ExtendedFaresComparingResult.BOTH
            ):
                necessary_to_insert_next_fare = True

            if (
                compare_result == ExtendedFaresComparingResult.SECOND or
                compare_result == ExtendedFaresComparingResult.ANY
            ):
                good_fares_indices_to_remove.append(good_fare_index)

        if not necessary_to_insert_next_fare and len(good_fares) > 0:
            return

        for good_fare_index in good_fares_indices_to_remove:
            good_fares.pop(good_fare_index, None)
        good_fares[next_fare_index] = next_fare

    def _choose_good_fares(self, extended_fares):
        # type: (Dict[int, Any]) -> Dict[int, Any]

        # Method scans all extended fares and returns concise set of the best.
        # Amount of best fares is an arbitrary number we can't predict before method starts.

        good_fares = {}

        for index, fare in extended_fares.iteritems():
            self._update_good_fares(good_fares, index, fare)

        return good_fares

    def _generate_fares_order_for_kateov_sort(self, good_fares_dictionary, all_fares_count):
        # type: (Dict[int, Any], int) -> Iterable[int]

        for good_fare_index in sorted(good_fares_dictionary.keys()):
            yield good_fare_index

        for rest_fare_index in xrange(all_fares_count):
            if rest_fare_index not in good_fares_dictionary:
                yield rest_fare_index

    def kateov_sort(self, search_result):
        """
        Experimental sort offered by kateov@.
        """

        fares_dict = dict(enumerate(search_result['fares']))

        extended_fares = self._extend_fares(search_result['fares'], search_result['flights'])
        extended_fares_dict = dict(enumerate(extended_fares))

        good_fares = self._choose_good_fares(extended_fares_dict)

        sorted_fares = [fares_dict[i] for i in self._generate_fares_order_for_kateov_sort(good_fares, len(fares_dict))]
        sorted_fares = self._place_cheapest_fare_on_first_position(sorted_fares)

        return self._search_result_with_new_fares(search_result, sorted_fares)

    def sort(self, search_result, query, sort_name):
        # type: (Any, Query, str) -> Any

        if sort_name == BigBeautySortings.SORTED_BY_PRICE:
            return self.sort_by_price(search_result)

        if sort_name == BigBeautySortings.CONTROL_WITH_WEEKDAYS:
            return self.control_with_weekdays(search_result, query)

        if sort_name == BigBeautySortings.FRONT_SORT:
            return self.front_sort(search_result, query)

        if sort_name == BigBeautySortings.KATEOV_SORT:
            return self.kateov_sort(search_result)

        raise Exception('Unsupported custom sort: %s' % sort_name)
