# -*- coding: utf-8 -*-

from typing import Any, Set
from datetime import datetime


_EPS_SHIFT = 5


class ExtendedFaresComparingResult(object):
    FIRST = 1
    SECOND = 2
    BOTH = 3
    ANY = 4

    @classmethod
    def possible_results(cls):
        # type: () -> Set[int]

        return {cls.FIRST, cls.SECOND, cls.BOTH, cls.ANY}


def sign(x):
    if x > 0:
        return 1

    if x < 0:
        return -1

    return 0


class ExtendedFaresComparator(object):
    def _difference_in_minutes(self, first_time, second_time):
        # type: (datetime, datetime) -> int

        return int((first_time - second_time).total_seconds() / 60)

    def compare(self, first, second):
        # type: (Any, Any) -> int

        # Comparator takes two extended fares and try to decide
        #   whether one of the fares is much better than another.
        # If one of the fares is much better than method returns
        #   ExtendedFaresComparingResult.FIRST and ExtendedFaresComparingResult.SECOND accordingly.
        # ExtendedFaresComparingResult.BOTH is returned if both fares are pretty different and both valuable for us.
        # ExtendedFaresComparingResult.ANY is returned
        #   if fares are pretty identical and we can be substituted by each other.

        # Step 1. Here we compare two equal fares by popularity.

        if (
            first['duration'] == second['duration'] and
            first['start_time'] == second['start_time'] and
            first['transfers_count'] == second['transfers_count'] and
            first['tariff']['value'] == second['tariff']['value'] and
            first['discomfort_level'] == second['discomfort_level']
        ):
            if first['popularity'] < second['popularity']:
                return ExtendedFaresComparingResult.SECOND

            if first['popularity'] > second['popularity']:
                return ExtendedFaresComparingResult.FIRST

            return ExtendedFaresComparingResult.BOTH

        # Step 2. If we see that these fares do not overlap each other and happen in different time
        #    then both fares are important for us.

        wE = (first['duration'] + second['duration']) >> (_EPS_SHIFT + 1)
        start_time_difference = self._difference_in_minutes(second['start_time'], first['start_time'])
        end_time_difference = self._difference_in_minutes(second['end_time'], first['end_time'])

        if sign(start_time_difference) == sign(end_time_difference) and abs(start_time_difference + end_time_difference) > wE:
            return ExtendedFaresComparingResult.BOTH

        # Step 3.

        transfers_count_difference = second['transfers_count'] - first['transfers_count']
        duration_difference = second['duration'] - first['duration']
        price_difference = second['tariff']['value'] - first['tariff']['value']
        discomfort_level_difference = second['discomfort_level'] - first['discomfort_level']

        if (
            transfers_count_difference >= 0 and
            duration_difference >= 0 and
            price_difference >= 0 and
            discomfort_level_difference >= 0
        ):
            return ExtendedFaresComparingResult.FIRST

        if (
            transfers_count_difference <= 0 and
            duration_difference <= 0 and
            price_difference <= 0 and
            discomfort_level_difference <= 0
        ):
            return ExtendedFaresComparingResult.SECOND

        # Step 4.

        transfers_count_differs_a_bit = second['transfers_count'] == first['transfers_count']
        duration_differs_a_bit = abs(duration_difference) <= wE
        prices_differs_a_bit = abs(price_difference) <= (int(second['tariff']['value'] + first['tariff']['value']) >> (_EPS_SHIFT + 1))
        discomfort_level_differs_a_bit = (
            abs(discomfort_level_difference) <= max((second['discomfort_level'] + first['discomfort_level']) >> (_EPS_SHIFT - 1), 10)
        )

        if (
            transfers_count_differs_a_bit and
            duration_differs_a_bit and
            prices_differs_a_bit and
            discomfort_level_differs_a_bit
        ):
            def relative_badness(fare):
                return fare['transfers_count'] + fare['duration'] + fare['tariff']['value'] + fare['discomfort_level']

            if relative_badness(first) < relative_badness(second):
                return ExtendedFaresComparingResult.FIRST

            return ExtendedFaresComparingResult.SECOND

        # Step 5

        if (
            (transfers_count_difference >= 0 or transfers_count_differs_a_bit) and
            (duration_difference >= 0 or duration_differs_a_bit) and
            (price_difference >= 0 or prices_differs_a_bit) and
            (discomfort_level_difference >= 0 or discomfort_level_differs_a_bit)
        ):
            return ExtendedFaresComparingResult.FIRST

        if (
            (transfers_count_difference <= 0 or transfers_count_differs_a_bit) and
            (duration_difference <= 0 or duration_differs_a_bit) and
            (price_difference <= 0 or prices_differs_a_bit) and
            (discomfort_level_difference <= 0 or discomfort_level_differs_a_bit)
        ):
            return ExtendedFaresComparingResult.SECOND

        return ExtendedFaresComparingResult.BOTH
