# -*- coding: utf-8 -*-
import copy
import inspect
import json
from itertools import product

from functools import partial, wraps
from importlib import import_module
from logging import getLogger
from operator import attrgetter

import gevent
from django.conf import settings
from django.utils.encoding import force_unicode
from django.utils.functional import cached_property
from requests.exceptions import RequestException

from travel.avia.library.python.avia_data.libs.currency.rates.caching import CurrencyRatesCache
from travel.avia.library.python.common.models.partner import DohopVendor, Partner
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.currency_converter import ConverterCache

from travel.avia.ticket_daemon.ticket_daemon.api.result import set_partners_statuses, Statuses
from travel.avia.ticket_daemon.ticket_daemon.api.flights import FlightFabric
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.lib.feature_flags import multisearch
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import requests_response_params
from travel.avia.ticket_daemon.ticket_daemon.lib.timelines import QueryTimeLines
from travel.avia.ticket_daemon.ticket_daemon.lib.timer import Timer
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers import search_result_logger, wizard_search_result_logger
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.tskv_logger import TskvLogger
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import dict_merge, group_by, log_filter
from travel.avia.ticket_daemon.ticket_daemon.lib import feature_flags
from travel.avia.ticket_daemon.ticket_daemon.daemon.async_helpers import Harvester
import travel.avia.ticket_daemon.ticket_daemon.daemon.big_beauty_collector as bbc
from travel.avia.ticket_daemon.ticket_daemon.daemon.importer_dialog import ImporterDialog
from travel.avia.ticket_daemon.ticket_daemon.daemon.timeout_utils import consume_with_whole_timeout
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import BadPartnerResponse, closing_db, TimedChunk
from travel.avia.ticket_daemon.ticket_daemon.daemon.variants_saver import VariantsSaver
from travel.avia.ticket_daemon.ticket_daemon.daemon.response_collector import ResponseCollector
from travel.avia.ticket_daemon.ticket_daemon.settings import WRITE_IN_PARTNER_CACHE

log = getLogger(__name__)
log_flow = getLogger('flow')
log_partners_errors = getLogger('partners_errors')
log_error_responses = getLogger('error_responses')

log_yt_partners_query = TskvLogger(
    name='yt.partners_query',
    environment=environment,
    tskv_format='rasp-partners-query-log',
)


def make_importer_query(query, importer):
    query = copy.copy(query)
    query.importer = importer

    return query


def create_multisearch(fn):
    @wraps(fn)
    def multisearch_wrapper(q):
        """
        :param ticket_daemon.api.query.Query q:
        :return:
        """
        for point_from, point_to in product(q.iatas_from, q.iatas_to):
            q.iata_from = point_from
            q.iata_to = point_to
            variants = fn(q)
            if inspect.isgenerator(variants):
                for chunk in variants:
                    yield chunk
            else:
                yield variants

    return multisearch_wrapper


def chunkify_queryfun(fn):
    @wraps(fn)
    def chunked_wrapper(q):
        variants = fn(q)

        if inspect.isgenerator(variants):
            for chunk in variants:
                yield chunk

        else:
            yield variants

    return chunked_wrapper


class HarvesterClosingDb(Harvester):
    def _run(self):
        try:
            with closing_db():
                super(HarvesterClosingDb, self)._run()
        except Exception:
            log.exception('HarvesterClosingDb worker fail')
            raise


class Importer(object):
    ENABLE_MULTI_SEARCH = True

    def __init__(self, code, partners, query, response_collector, flight_fabric, big_beauty_collectors,
                 custom_store_time=None):
        self.timer = Timer()
        self.code = code
        self.partners = partners
        self.response_collector = response_collector
        self.flight_fabric = flight_fabric  # type: FlightFabric
        self.big_beauty_collectors = big_beauty_collectors
        self.savers_by_partner = {}
        self.q = make_importer_query(query, self)
        self.custom_store_time = custom_store_time

    @cached_property
    def key(self):
        return '%s[%s]' % (
            self.code, ','.join([p.code for p in self.partners])
        )

    @cached_property
    def id_msg(self):
        return u'%s %s' % (self.q.qid_msg, self.key)

    @cached_property
    def currency_rates(self):
        rates_caches = filter(None, [
            ConverterCache.load(self.q.national_version),
            CurrencyRatesCache.load(self.q.national_version),
        ])
        if not rates_caches:
            return None

        return dict_merge(*[cache.rates for cache in rates_caches])

    @classmethod
    def get_importer_code_for_partner(cls, partner):
        if partner.__class__.__name__ == 'DohopVendor':
            return partner.query_module_name or 'dohop'
        else:
            # Подходит и для AmadeusMerchant
            return partner.query_module_name or partner.code

    @classmethod
    def gen_for_partners(cls, partners, query):
        response_collector = ResponseCollector(query)
        big_beauty_collectors = cls._build_big_beauty_collectors(query)
        flight_fabric = FlightFabric()

        by_importer_code = group_by(partners, cls.get_importer_code_for_partner)

        for importer_code, importer_partners in by_importer_code.iteritems():
            try:
                importer_cls = cls.get_importer_cls_by_code(importer_code)
                yield importer_cls(
                    code=importer_code,
                    partners=importer_partners,
                    query=query,
                    response_collector=response_collector,
                    big_beauty_collectors=big_beauty_collectors,
                    flight_fabric=flight_fabric,
                    custom_store_time=query.meta['custom_store_time'],
                )
            except Exception:
                log.exception(u'Error creating importer %r', importer_code)
                continue

    @classmethod
    def _build_big_beauty_collectors(cls, query):
        wizard_caches = query.meta.get('wizard_caches', [])
        collectors = []
        if len(wizard_caches) == 0:
            collectors.append(cls._build_big_beauty_collector(query, is_experimental=False))
            if feature_flags.store_experimental_wizard_results():
                collectors.append(cls._build_big_beauty_collector(query, is_experimental=True))
            cls._add_collectors_by_partner(query, collectors)
            return collectors
        if 'wizard_results' in wizard_caches:
            collectors.append(cls._build_big_beauty_collector(query, is_experimental=False))
        if feature_flags.store_experimental_wizard_results() and 'wizard_results_experimental' in wizard_caches:
            collectors.append(cls._build_big_beauty_collector(query, is_experimental=True))
        cls._add_collectors_by_partner(query, collectors)
        return collectors

    @staticmethod
    def _build_big_beauty_collector(query, is_experimental):
        wizard_stored_results = bbc.get_stored_search_result(query, is_experimental)
        big_beauty_collector = bbc.BigBeautyCollector(query, wizard_stored_results, wizard_search_result_logger, is_experimental)

        if wizard_stored_results.version == 0:
            big_beauty_collector.store()
        return big_beauty_collector

    @staticmethod
    def _build_big_beauty_collector_by_partner(query, partner_code, wizard_stored_result_by_partner):
        big_beauty_collector_by_partner = bbc.BigBeautyCollectorByPartner(
            query, wizard_stored_result_by_partner, wizard_search_result_logger, partner_code,
        )

        return big_beauty_collector_by_partner

    @classmethod
    def _add_collectors_by_partner(cls, query, collectors):
        if not WRITE_IN_PARTNER_CACHE():
            return
        try:
            partner_codes = [p.code for p in query.partners]
            wizard_stored_results_by_partner = bbc.get_stored_search_results_by_partner(query, partner_codes)
            for partner_code in partner_codes:
                try:
                    collectors.append(cls._build_big_beauty_collector_by_partner(query, partner_code, wizard_stored_results_by_partner.get(partner_code)))
                except Exception as exc:
                    log.exception(u'Exception %r on create BigBeautyCollectorByPartner %s', exc, partner_code)
        except Exception as exc:
            log.exception(u'Exception %r on create BigBeautyCollectorByPartner', exc)

    @classmethod
    def get_importer_cls_by_code(cls, code):
        if code.startswith('dohop'):
            return DohopImporter
        if code.startswith('amadeus'):
            return AmadeusImporter
        if code == 'pobeda':
            return PobedaImporter
        else:
            return Importer

    def assign_partners_to_variants(self, variants):
        if len(self.partners) == 1:
            partner = self.partners[0]

            for v in variants:
                v.partner = partner
                v.partner_code = partner.code

        else:
            partners_by_code = {p.code: p for p in self.partners}

            for v in variants:
                try:
                    p_code = v.partner_code

                except AttributeError:
                    raise Exception(
                        'Variant should have attribute partner_code '
                        'for many-partners importer'
                    )

                # Ключ должен быть в словаре. Иначе это ошибка модуля импорта
                if p_code not in partners_by_code:
                    raise Exception('Unknown partner_code: %r' % p_code)

                v.partner = partners_by_code[p_code]

    @property
    def dialog(self):
        return ImporterDialog(self.q, self.code)

    @classmethod
    def prepare_chunked_queryfun(cls, code, q):
        try:
            query_module = ModuleImporter.import_module('travel.avia.ticket_daemon.ticket_daemon.partners.%s' % code)
        except Exception as exc:
            raise QueryIsNotValid('Can not import %s query module. %r' % (code, exc))

        if not query_module:
            raise QueryIsNotValid('No query_module')

        q.validate(query_module)

        if 'query' not in dir(query_module):
            raise QueryIsNotValid('No "query" method in query_module')

        queryfun = query_module.query
        if not queryfun:
            raise QueryIsNotValid('No query_function')

        if multisearch() and cls.ENABLE_MULTI_SEARCH:
            return chunkify_queryfun(create_multisearch(queryfun))
        else:
            return chunkify_queryfun(queryfun)

    def do_import(self):
        try:
            self._do_import()
        except BaseException:
            self._skip_import()
            raise

    def _do_import(self):
        log_flow.info(u'%s do_import', self.id_msg)

        try:
            chunked_queryfun = self.prepare_chunked_queryfun(self.code, self.q)

        except QueryIsNotValid as err:
            log_flow.info(u'%s query_invalid %s', self.id_msg, _try_repr_exception(err))
            self.yt_write(u'query_invalid', {'importer': self.code})
            self.dialog.write_dialog_status(Statuses.SKIP, _try_repr_exception(err))
            self._stop_querying()
            return

        # Сообщаем остальным, что запрос уже идет
        set_partners_statuses(
            self.q, self.partners, Statuses.QUERYING,
            custom_store_time=settings.QUERY_PARTNERS_TIMEOUT
        )
        gevent.spawn(self._start_querying, chunked_queryfun)

    def _start_querying(self, chunked_queryfun):
        self.dialog.write_dialog_status(Statuses.QUERYING)

        log_flow.info(u'%s querying', self.id_msg)

        self.savers_by_partner = {
            partner: VariantsSaver(self, partner, search_result_logger)
            for partner in self.partners
        }

        def each_saver(fn):
            for saver in self.savers_by_partner.values():
                try:
                    fn(saver)
                except Exception as e:
                    log.warning('Saver %r method Exception: %r', saver, e)

        self.q.timeline.event(
            QueryTimeLines.events.start_import, {'importer': self.code}
        )
        self.q.timeline_external.event(
            QueryTimeLines.events.start_import, {'importer': self.code}
        )

        with HarvesterClosingDb() as harvester:
            try:
                chunks_gen = consume_with_whole_timeout(
                    settings.QUERY_PARTNERS_TIMEOUT,
                    chunked_queryfun(self.q)
                )

                for variants_chunk in chunks_gen:
                    harvester.put_job(partial(self.got_chunk, variants_chunk))

            # Тут надо ловить все
            except BaseException as e:
                each_saver(lambda s: s.set_failure(e))

                try:
                    raise
                except BadPartnerResponse as e:
                    self.got_bad_partner_response(e)

                except gevent.Timeout:
                    self.got_timeout()

                except RequestException as e:
                    self.got_failure(e)

                except Exception as e:
                    log_partners_errors.exception(self.id_msg)
                    self.got_failure(e)

        # Done all harvester jobs

        each_saver(lambda s: s.finalize())

        self.dialog.write_dialog_exchanges()

    def _skip_import(self):
        for partner in self.partners:
            self.response_collector.skip_partner(partner)
            for bbc_collector in self.big_beauty_collectors:
                bbc_collector.partner_done(partner)

    def _stop_querying(self):
        self._skip_import()

        # это место можно с оптимизировать, группировкой по store_time
        # и затем сделать массовый set, но пока не будем этого делать
        for partner in self.partners:
            VariantsSaver(self, partner, search_result_logger).store_skip()

    def got_chunk(self, chunk):
        try:
            self._got_chunk(chunk)
        except Exception as e:
            self.dialog.write_dialog_exception()
            log.exception('got_chunk')
            log_flow.warning(u'%s got_chunk_error %s', self.id_msg, _try_repr_exception(e))
            self.yt_write('got_chunk_error')

    def _got_chunk(self, chunk):
        assert isinstance(chunk, TimedChunk)

        fetch_time = self.timer.get_elapsed_seconds()

        query_time = chunk.query_time or fetch_time
        variants = chunk.variants

        chunk_process_timer = Timer()

        variants = preprocess_variants(variants, self.id_msg)

        self.assign_partners_to_variants(variants)
        # Отфильтруем потому что может быть не заполнен v.partner
        variants = list(log_filter(lambda v: hasattr(v, 'partner') and v.partner is not None, variants, logger=log))

        variants_by_partner = group_by(variants, attrgetter('partner'))

        for partner, partner_variants in variants_by_partner.iteritems():
            saver = self.savers_by_partner[partner]
            saver.add_variants_chunk(partner_variants, query_time)

        self.yt_write('got_reply', extra={
            'query_time': query_time * 1000,  # В yt - в миллисекундах
            'fetch_time': fetch_time * 1000,  # В yt - в миллисекундах
            'packing_time': chunk_process_timer.get_elapsed_seconds() * 1000,
            'variants_len': str(len(variants))
        })

    def got_bad_partner_response(self, e):
        log_flow.warning(u'%s response_error %s', self.id_msg, _try_repr_exception(e))
        extra = {'errors': e.errors} if e.errors else {}
        self.yt_write('response_error', extra=extra)
        log_error_responses.info(
            u'%s %s', e.partner_code,
            json.dumps(requests_response_params(e.response))
        )
        self.dialog.write_dialog_status('BadPartnerResponse')

    def got_failure(self, e):
        log_flow.warning(u'%s got_failure %s', self.id_msg, _try_repr_exception(e))
        self.yt_write('got_failure')
        self.dialog.write_dialog_exception()

    def got_timeout(self):
        log_flow.warning(u'%s timeout', self.id_msg)
        self.yt_write('timeout')
        self.dialog.write_dialog_status('timeout')

    def yt_write(self, status, extra=None):
        data = {
            'importer': self.key,
            'status': status,
            'qid': self.q.meta.get('base_qid') or self.q.id,
            'init_id': self.q.id,
        }

        if extra:
            data.update(extra)

        if 'query_time' not in data:
            data['query_time'] = self.timer.get_elapsed_seconds() * 1000

        log_yt_partners_query.log(data)

    def __repr__(self):
        return '<%s: %r>' % (self.__class__.__name__, self.code)


def _try_repr_exception(e):
    try:
        return force_unicode(repr(e))
    except Exception:
        return force_unicode(e)


# Хак для U6 как партнёра
Partner.merchant_id = property(lambda p: p.code)


class AmadeusImporter(Importer):
    def assign_partners_to_variants(self, variants):
        merchants_by_id = {
            p.merchant_id.lower(): p
            for p in self.partners
            if p.merchant_id
        }

        for v in variants:
            try:
                v.partner = merchants_by_id[v.amadeus_merchant_id]
                v.partner_code = v.partner.code

            except KeyError:
                log.info('Skip Amadeus variant with merchant id: %s', v.amadeus_merchant_id)


class DohopImporter(Importer):
    ENABLE_MULTI_SEARCH = False

    def assign_partners_to_variants(self, variants):
        vendors_by_id = {p.dohop_id: p for p in self.partners if isinstance(p, DohopVendor)}
        partners_by_code = {p.code: p for p in self.partners if isinstance(p, Partner)}

        for v in variants:
            try:
                if v.partner_code in partners_by_code:
                    v.partner = partners_by_code[v.partner_code]
                else:
                    v.partner = vendors_by_id[v.dohop_vendor_id]

                v.partner_code = v.partner.code

            except KeyError:
                # Дохоп может в любой момент добавить нового партнера
                log.warning('Skip Dohop variant with unknown vendor id: %s' % v.dohop_vendor_id)

    def _got_chunk(self, chunk):
        assert isinstance(chunk, TimedChunk)
        super(DohopImporter, self)._got_chunk(chunk)


class PobedaImporter(Importer):
    ENABLE_MULTI_SEARCH = False


class ModuleImporter(object):
    @staticmethod
    def import_module(module_name):
        try:
            return import_module(module_name)

        except ImportError:
            log.exception(
                u'Can not import partner query module %r' % module_name)


def preprocess_variants(variants, id_msg):
    for v in variants:
        for f in v.all_segments:
            try:
                f.preprocess()

            except Exception:
                log.exception(u'%s flight.preprocess', id_msg)

    return variants
