# -*- coding: utf-8 -*-
import json
import logging
import zlib
from traceback import format_exc

from travel.avia.ticket_daemon.ticket_daemon.api.cache import shared_cache
from travel.avia.ticket_daemon.ticket_daemon.api.result import Statuses
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import dict_merge, skip_None_values


log = logging.getLogger(__name__)


class ImporterDialog(object):
    def __init__(self, query, importer_code):
        self.q = query
        self.importer_code = importer_code
        self.key = '%s/track/%s' % (self.q.id, self.importer_code)

    @staticmethod
    def activate_query(q):
        q.trackers = {}

    @property
    def _is_active(self):
        return hasattr(self.q, 'trackers') and isinstance(self.q.trackers, dict)

    def write_dialog_exception(self):
        if not self._is_active:
            return

        self._write_dialog_error(format_exc())

    def _write_dialog_error(self, description):
        if not self._is_active:
            return

        self.write_dialog_status('error', description)

    def write_dialog_status(self, status, description=None):
        if not self._is_active:
            return

        log.info('write_dialog_status %s %s %r', self.key, status, description)
        self._write_dialog_data(dict_merge(
            {'status': status},
            skip_None_values({'description': description})
        ))

    def write_dialog_exchanges(self):
        if not self._is_active:
            return

        try:
            tracker = self.q.trackers.values()[0]

        except IndexError:
            self.write_dialog_status('no_trackers')
            return

        self._write_dialog_data({
            'exchanges': [
                {
                    'id': exchange.id,
                    'title': exchange.title,
                    'events': [
                        {
                            'code': event.code,
                            'content': event.content,
                        }
                        for event in exchange.events
                    ],
                }
                for exchange in tracker.exchanges
            ]
        })

    def _write_dialog_data(self, data):
        if not self._is_active:
            return

        try:
            result = json.dumps(data)

        except Exception:
            self.write_dialog_exception()

        else:
            self._write_dialog_result(result)

    def _write_dialog_result(self, value):
        if not self._is_active:
            return

        compressed_value = zlib.compress(value) if value else value
        log.info('try to store %s data size %s, compressed size %s', self.key, len(value), len(compressed_value))
        store_time = 1800  # На полчаса
        shared_cache.set(self.key, compressed_value, store_time)

    @staticmethod
    def get_results_by_key(key):
        value = shared_cache.get(key)
        if value:
            value = zlib.decompress(value)

        result = {}

        if not value:
            result['status'] = Statuses.NONE

        elif value == Statuses.QUERYING:
            result['status'] = Statuses.QUERYING

        else:
            result['status'] = Statuses.DONE

            try:
                result.update(json.loads(value))

            except Exception:
                result.update({
                    'status': 'corrupt',
                    'description': format_exc(),
                })

        return result
