# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals

import ujson
from datetime import datetime
from django.conf import settings
from logging import getLogger

from travel.avia.library.python.common.utils import environment
from travel.avia.ticket_daemon.ticket_daemon.api.result import cache_backends
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.processed_qid_logger import processed_qid_logger
from travel.avia.ticket_daemon.ticket_daemon.daemon.tasks import query_aggregate_min_prices_task
from travel.avia.ticket_daemon.ticket_daemon.daemon.logbroker import get_processed_qid_writer
from travel.avia.ticket_daemon.ticket_daemon.lib.timelines import QueryTimeLines

log = getLogger(__name__)
all_empty_variants_query_log = getLogger('all_empty_variants_query')
empty_variants_query_log = getLogger('empty_variants_query')
log_flow = getLogger('flow')


class ResponseCollector(object):
    def __init__(self, q):
        self.q = q
        self.partners = {p.code: p for p in q.partners}
        self.partners_variants = {}
        self.partners_ready = {}
        self.partners_empty = set()

    def skip_partner(self, partner):
        self.partners.pop(partner.code, None)

    def partner_done(self, partner):
        if self.partners_ready.get(partner.code):
            return

        self.partners_ready[partner.code] = True

        if not self.partners_variants.get(partner.code):
            log_empty_variants(self.q, partner)
            self.partners_empty.add(partner.code)

        self.check_fullness(current_partner=partner)

    def add_response(self, partner, result):
        if self.partners_ready.get(partner.code):
            raise Exception('Do not add_response after partner_done')
        self.partners_variants.setdefault(partner.code, []).extend(result.variants)
        self.check_fullness(current_partner=partner)

    def partner_done_with_failure(self, partner, code):
        self.partner_done(partner)

        if not self.partners_variants.get(partner.code):
            log_empty_variants(self.q, partner, failure=code)

    def check_fullness(self, current_partner):
        if len(self.partners_variants) == 1:
            self.q.timeline.event(
                QueryTimeLines.events.first_partner,
                {'code': current_partner.code}
            )

        self.check_all_replied(current_partner)

    def check_all_replied(self, current_partner):
        # вызвать колбэк, если все кончили
        if not self.all_replied:
            return

        self.q.timeline.event(
            QueryTimeLines.events.last_partner,
            {'code': current_partner.code}
        )

        if self.all_empty:
            log_all_empty_variants(self.q)
        if settings.AGGREGATE_MIN_PRICES:
            self._aggregate_min_price()

        self._remove_partners_with_empty_responses()

        self._send_processed_qid_to_logbroker(self.q.id)

    def _aggregate_min_price(self):
        try:
            if settings.AGGREGATE_MIN_PRICES_SYNC:
                query_aggregate_min_prices_task(self.q.id)
            else:
                query_aggregate_min_prices_task.apply_async(
                    (self.q.id,), expires=60 * 1
                )

        except Exception:
            log.exception('Error set task query_aggregate_min_prices_task')
        else:
            log.debug('Set task query_aggregate_min_prices_task')

    def _send_processed_qid_to_logbroker(self, qid):
        if len(self.partners_ready) > 0:
            try:
                if settings.LOGBROKER_PROCESSED_QID_WRITE_TO_FILE:
                    processed_qid_logger.log(qid)
                if settings.LOGBROKER_PROCESSED_QID_WRITE_TO_LB:
                    data = ujson.dumps({'qid': qid, 'unixtime': environment.unixtime()}).encode('utf-8')
                    get_processed_qid_writer().write(data)
            except:
                log.exception('Failed to send processed qid to logbroker')

    def _remove_partners_with_empty_responses(self):
        if not self.partners_empty:
            return
        log.info('Removing partners with empty responses: %s for query id %s', self.partners_empty, self.q.id)
        try:
            cache_backends.ydb_cache.remove_many(self.q, self.partners_empty)
        except Exception, exc:
            log.exception('Exception %r on removing partners with empty responses', exc)

    @property
    def all_replied(self):
        u""" Все ли ответили """
        return all([p.code in self.partners_ready for p in self.partners.values()])

    @property
    def all_empty(self):
        u""" Являются ли все ответы пустыми """
        return all([not self.partners_variants.get(p.code) for p in self.partners.values()])


def log_empty_variants(q, partner, failure=None):
    if failure:
        if hasattr(failure, 'status'):
            failure_str = str(failure.status)

        else:
            failure_str = repr(failure)

    else:
        failure_str = '200'

    empty_variants_query_log.info(
        u'datetime=%s '
        u'status=%s '
        u'partner=%s '
        u'from=%s '
        u'to=%s '
        u'date_forward=%s '
        u'date_backward=%s '
        u'adults=%s '
        u'children=%s '
        u'infants=%s '
        u'class=%s' % (
            str(datetime.now().strftime('%Y-%m-%dT%H:%M:%S')),
            failure_str,
            partner.code,
            q.iata_from,
            q.iata_to,
            str(q.date_forward),
            str(q.date_backward),
            q.passengers.get('adults', 0),
            q.passengers.get('children', 0),
            q.passengers.get('infants', 0),
            q.klass
        )
    )


def log_all_empty_variants(q):
    log_flow.info(u'%s all_empty', q.qid_msg)

    all_empty_variants_query_log.info(
        u'from=%s '
        u'to=%s '
        u'date_forward=%s '
        u'date_backward=%s '
        u'adults=%s '
        u'children=%s '
        u'infants=%s '
        u'class=%s' % (
            q.iata_from,
            q.iata_to,
            str(q.date_forward),
            str(q.date_backward),
            q.passengers.get('adults', 0),
            q.passengers.get('children', 0),
            q.passengers.get('infants', 0),
            q.klass
        )
    )
