# -*- coding: utf-8 -*-
from contextlib import contextmanager
from functools import partial, wraps

import gevent
from decorator import decorator
from django import db

from travel.avia.library.python.common.utils.exceptions import SimpleUnicodeException


class TimedChunk(object):
    def __init__(self, variants, query_time):
        self.variants = variants
        self.query_time = query_time

    def __iter__(self):
        return iter(self.variants)


class PartnerErrorTypes(object):
    ERROR = 'ERROR'  # Не привязан к конкретной ошибке, может возвращаться в ряде случаев
    SYSTEM_ERROR = 'SYSTEM_ERROR'  # Внутренняя ошибка работы системы, при получении рекомендуем обращаться в службу техподдержки
    ACCESS_DENIED = 'ACCESS_DENIED'  # 'Ошибка уровня доступа к чему-либо в системе
    AUTHORIZATION_ERROR = 'AUTHORIZATION_ERROR'  # Ошибка авторизации
    IVALID_REQUEST_STRUCTURE = 'IVALID_REQUEST_STRUCTURE'  # Ошибка в структуре и параметрах запроса к серверу
    SABRE_GDS_ERROR = 'SABRE_GDS_ERROR'  # Необрабатываемая ошибка от поставщика Sabre
    GDS_CONNECTION_ERROR = 'GDS_CONNECTION_ERROR'  # Ошибка связи с поставщиком
    AMADEUS_GDS_ERROR = 'AMADEUS_GDS_ERROR'  # Необрабатываемая ошибка от поставщика Amadeus
    SIRENA_GDS_ERROR = 'SIRENA_GDS_ERROR'  # Необрабатываемая ошибка от поставщика SIRENA-TRAVEL
    GALILEO_GDS_ERROR = 'GALILEO_GDS_ERROR'  # Необрабатываемая ошибка от поставщика Travelport
    SITA_GABRIEL_GDS_ERROR = 'SITA_GABRIEL_GDS_ERROR'  # Необрабатываемая ошибка от поставщика SITA Gabriel
    VIP_SERVICE_GDS_ERROR = 'VIP_SERVICE_GDS_ERROR'  # Необрабатываемая ошибка от поставщика ТАИС - SIG
    NI_GDS_ERROR = 'NI_GDS_ERROR'  # Необрабатываемая ошибка от поставщика Nemo Inventory
    PEGASUS_GDS_ERROR = 'PEGASUS_GDS_ERROR'  # Необрабатываемая ошибка от поставщика Pegasus
    TF_GDS_ERROR = 'TF_GDS_ERROR'  # Необрабатываемая ошибка от поставщика Travelfusion
    SEARCHES_LIMIT_ERROR = 'SEARCHES_LIMIT_ERROR'  # Превышен лимит на количество поисков
    SEARCHES_PER_TICKET_LIMIT_ERROR = 'SEARCHES_PER_TICKET_LIMIT_ERROR'  # Превышено допустимое количество поисков на выписку
    TARIFF_RULES_PARSING_ERROR = 'TARIFF_RULES_PARSING_ERROR'  # Ошибка парсинга тарифных правил
    NETWORK_ERROR = 'NETWORK_ERROR'  # Сетевая ошибка при обращении к поставщику
    USER_CONFIGURATION_ERROR = 'USER_CONFIGURATION_ERROR'  # Некорретная конфигурация пользователя, обратитесь в службу техподдержки
    DATE_IN_THE_PAST = 'DATE_IN_THE_PAST'  # Передана дата в прошлом
    CYRILLIC_CODE = 'CYRILLIC_CODE'  # Передан код города/аэропорта с использованием кириллицы


class BadPartnerResponse(SimpleUnicodeException, gevent.GreenletExit):
    def __init__(self, partner_code, response, errors=None):
        self.partner_code = partner_code
        self.response = response
        self.errors = errors

        msg = '[%s] (%s)' % (
            response.status_code,
            response.reason.encode('string-escape') if response.reason else ''
        )

        super(BadPartnerResponse, self).__init__(msg)


def async_sleep(val):
    return gevent.sleep(val)


def sleep_every(generator, count=500, sleep_time=0):
    i = 0
    for item in generator:
        i += 1
        if i % count == 0:
            gevent.sleep(sleep_time)

        yield item


def spawn_worker(*args, **kwargs):
    return gevent.spawn(*args, **kwargs)


def gen_workers_results(workers):
    for worker in workers:
        result = worker.get()

        if isinstance(result, BadPartnerResponse):
            raise result

        yield result


def collect_workers_results(workers):
    return list(gen_workers_results(workers))


@contextmanager
def closing_db(force=False, logger=None):
    from django import db

    # В daemon.mysql_switcher.base.DatabaseWrapper
    # включен запрет работать с базой без этого контекста
    with db.connection.context():
        # if force or (
        #     # Обновлять соединение только во внешней обёртке
        #     ctx.level <= 1
        #     # Обновлять соединение только в 1/1000 случаев
        #     and random() < 1 / 1000.0
        # ):
        if force:
            db.reset_queries()
            if logger:
                logger.info('Connections are reset')
            try:
                yield
            finally:
                db.close_old_connections()
                if logger:
                    logger.info('Connections are closed')
        else:
            yield


@decorator
def use_db(func, *args, **kwargs):
    """Сохраняющий сигнатуру функции декоратор обычной функции"""
    with closing_db():
        return func(*args, **kwargs)


def with_closing_db(func):
    """В эту функцию можно передавать bound-методы"""
    @wraps(func)
    def wrapper(*args, **kwargs):
        with closing_db():
            return func(*args, **kwargs)
    return wrapper


@contextmanager
def allow_db_ctx(logger=None):
    with db.connection.context():
        yield


@decorator
def allow_db(func, *args, **kwargs):
    with allow_db_ctx():
        return func(*args, **kwargs)


def allow_db_in_method(func):
    """В эту функцию можно передавать bound-методы"""
    @wraps(func)
    def wrapper(*args, **kwargs):
        with allow_db_ctx():
            return func(*args, **kwargs)
    return wrapper


def logit(logger, fn):
    import logging
    log = logging.getLogger(__name__)

    @wraps(fn)
    def _logit(*args, **kwargs):
        try:
            return fn(*args, **kwargs)
        except Exception:
            try:
                raise
            finally:
                try:
                    logger.exception('Error in %r', fn)
                except Exception:
                    log.exception('Logging error')

    return _logit


def logit_with(logger_method, fn=None):
    if fn is None:
        # Act as parametrizable decorator
        return partial(_logit_with, logger_method)
    else:
        return _logit_with(logger_method, fn)


def _logit_with(logger_method, fn):
    import logging
    log = logging.getLogger(__name__)

    @wraps(fn)
    def _logit(*args, **kwargs):
        try:
            return fn(*args, **kwargs)
        except Exception as e:
            try:
                raise
            finally:
                try:
                    logger_method('Error in %r: %r', fn, e)
                except Exception:
                    log.exception('Logging error')

    return _logit


@decorator
def eat_exception(func, *args, **kwargs):
    """Игнорирует ошибки вызова функции, возвращает None"""
    try:
        return func(*args, **kwargs)
    except Exception:
        pass


def fill(obj, **kwargs):
    for k, v in kwargs.items():
        setattr(obj, k, v)
    return obj
