# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals

from logging import getLogger

import gevent

from travel.avia.ticket_daemon.ticket_daemon.api.result import (
    Result, Status, Statuses, MinPrice, prepare_variants_for_result,
)
from travel.avia.ticket_daemon.ticket_daemon.daemon.big_beauty_collector import BigBeautyCollectorByPartner
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import BadPartnerResponse
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_store_time_provider import partner_store_time_provider
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.variants_logger import variants_logger
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.variants_logger2 import VariantsLogger2

log = getLogger(__name__)
log_flow = getLogger('flow')

MIN_PRICE_STORE_TIME = 60 * 60 * 1  # один час в секундах


class VariantsSaver(object):
    def __init__(self, importer, partner, search_result_logger):
        self.importer = importer
        self.partner = partner
        self.q = importer.q
        self.id_msg = u'%s %s' % (self.q.qid_msg, self.partner.code)
        self._response_collector = importer.response_collector
        self._big_beauty_collectors = importer.big_beauty_collectors

        self.saved_results = []
        self.saved_variants = []
        self.was_failure = False
        self._exception = None
        self._import_time = None

        self.variants_logger2 = VariantsLogger2.create()
        self._search_result_logger = search_result_logger

    def add_variants_chunk(self, variants, query_time):
        try:
            self._add_variants_chunk(variants, query_time)
        except Exception:
            log.exception('ERROR add_variants_chunk')

    def _add_variants_chunk(self, variants, query_time):
        log_flow.info(u'%s got_reply %s', self.id_msg, len(variants))

        round_variants_tariffs(variants, self.id_msg)
        fill_national_tariff(
            variants, self.q.national_version, self.importer.currency_rates
        )
        variants = prepare_variants_for_result(
            self.q, self.partner, self.importer.currency_rates, variants
        )

        variants = unify_variants(variants, self.saved_variants)

        self._store_variants(variants, query_time)
        self._store_partner_min_price()

    def _store_variants(self, variants, query_time):
        log_flow.info(
            u'%s store_variants [%s] qt=%s lang=%s',
            self.id_msg, len(variants), query_time, self.q.lang
        )

        result = self._create_result(
            variants=variants,
            query_time=query_time
        )

        try:
            result.store()
            self._import_time = self.importer.timer.get_elapsed_seconds()
            log.info('YDB result saved')
        except Exception as exc:
            log.exception('Exception %r on saving to ydb', exc)

        for bbc_collector in self._big_beauty_collectors:
            try:
                bbc_collector.add(result)
            except Exception as exc:
                log.exception(u'Exception %r on adding wizard variants', exc)

        try:
            self._store_status(
                status_code=Statuses.QUERYING,
                all_variants_count=len(self.saved_variants) + len(result.variants),
            )

            self._response_collector.add_response(self.partner, result)

        except Exception as e:
            self._store_status(
                status_code=Statuses.FAIL,
                all_variants_count=0
            )

            log_flow.warning(u'%s store_result_bug %r', self.id_msg, e)

            log.exception(u'%s store_result_bug', self.id_msg)

        else:
            self.saved_results.append(result)
            self.saved_variants.extend(result.variants)
            log_flow.info(
                u'%s saved_variants [%s] qt=%s lang=%s',
                self.id_msg, len(result.variants), query_time, self.q.lang
            )

    def _store_partner_min_price(self):
        if self.saved_variants:
            try:
                MinPrice.store(
                    query=self.q,
                    partner=self.partner,
                    store_time=MIN_PRICE_STORE_TIME,
                    variants=self.saved_variants,
                    rates=self.importer.currency_rates,
                )
            except Exception:
                log.exception('Store min price error')

    def store_skip(self):
        self._store_status(
            status_code=Statuses.SKIP,
            all_variants_count=0
        )
        self._response_collector.partner_done_with_failure(self.partner, Statuses.SKIP)

    def _store_status(self, status_code, all_variants_count):
        Status(
            query=self.q,
            partner=self.partner,
            variants=[],  # Todo: не забыть выпилить
            store_time=partner_store_time_provider.get_status_time(
                partner=self.partner,
                custom_store_time=None
            ),
            status=status_code,
            all_variants_count=all_variants_count,
        ).store()

    def finalize(self):
        self._store_status(
            status_code=Statuses.FAIL if self.was_failure else Statuses.DONE,
            all_variants_count=len(self.saved_variants)
        )

        self.log_search_result()

        if self.saved_variants:
            variants_logger.log(
                query=self.q,
                partner=self.partner,
                partner_variants=self.saved_variants
            )

            self.variants_logger2.log(
                query=self.q,
                partner=self.partner,
                partner_variants=self.saved_variants
            )
        else:
            if not self.was_failure:
                log_flow.info(u'%s empty_response', self.id_msg)

        if self.was_failure and not self.saved_variants:
            self._response_collector.partner_done_with_failure(self.partner, Statuses.FAIL)
        else:
            self._response_collector.partner_done(self.partner)

        all_replied = False
        for bbc_collector in self._big_beauty_collectors:
            bbc_collector.partner_done(self.partner)
            if not isinstance(bbc_collector, BigBeautyCollectorByPartner) and bbc_collector.all_replied and not bbc_collector.is_experimental:
                all_replied = True

        if all_replied:
            for bbc_collector in self._big_beauty_collectors:
                if isinstance(bbc_collector, BigBeautyCollectorByPartner):
                    bbc_collector.store()

    def set_failure(self, exception):
        self.was_failure = True
        self._exception = exception

    def _create_result(self, variants, query_time):
        return Result(
            query=self.q,
            partner=self.partner,
            variants=variants,
            saved_variants=self.saved_variants,
            store_time=partner_store_time_provider.get_result_time(
                partner=self.partner,
                custom_store_time=self.importer.custom_store_time
            ),
            query_time=query_time
        )

    def log_search_result(self):
        errors = None
        if isinstance(self._exception, BadPartnerResponse):
            errors = self._exception.errors

        self._search_result_logger.log(
            self.q, self.partner.code, self.importer.code, self._flow_result(), self.q.meta['test_id'],
            variants_count=len(self.saved_variants),
            query_time=self._import_time,
            errors=errors,
        )

    def _flow_result(self):
        if not self.was_failure:
            if self.saved_variants:
                result = 'got_reply'
            else:
                result = 'empty_response'
        elif isinstance(self._exception, BadPartnerResponse):
            result = 'response_error'
        elif isinstance(self._exception, gevent.Timeout):
            result = 'timeout'
        else:
            result = 'failure'
        return result


def unify_variants(variants, previous):
    u"""
    Отфильтровать варианты, оставить по одной с багажом и без,
     самой минимальной, цене на каждый рейс от партнера,
     для каждого тарифа (fare_code).
    """

    minimal_tariffs = {}

    for v in previous:
        key = v.by_flights_key, v.by_fare_code, v.with_baggage
        if key not in minimal_tariffs:
            minimal_tariffs[key] = v.tariff
        else:
            if v.tariff < minimal_tariffs[key]:
                minimal_tariffs[key] = v.tariff

    filtered_variants = {}

    for v in variants:
        key = v.by_flights_key, v.by_fare_code, v.with_baggage

        if key not in minimal_tariffs:
            minimal_tariffs[key] = v.tariff
            filtered_variants[key] = v
        else:
            if v.tariff < minimal_tariffs[key]:
                minimal_tariffs[key] = v.tariff
                filtered_variants[key] = v

    return filtered_variants.values()


def round_variants_tariffs(variants, id_msg):
    for v in variants:
        try:
            rounded_value = round(v.tariff.value)
            if rounded_value != v.tariff.value:
                log.debug('%s Variant tariff: %r %r', id_msg, v.tariff, v)
            v.tariff.value = rounded_value

        except Exception as e:
            log.error('%s Cannot round variant tariff value: %r %r', id_msg, v, e)


def fill_national_tariff(variants, national_version, currency_rates):
    """Заполняем national_tariff лучше явно, а не в MinPrice
    :type variants: list of ticket_daemon.api.flights.Variant
    :param basestring national_version:
    :param dict currency_rates:
    :return:
    """
    for variant in variants:
        try:
            variant.national_tariff = Price.convert_to_national(
                variant.tariff, national_version, currency_rates
            )
        except ValueError as e:
            variant.national_tariff = None
            log.warning('Drop %s: %s', variant, e)
