import re

from typing import Optional

from travel.avia.library.python.avia_data.models import CompanyTariff

KEY_PARTS = re.compile(r'((\d{0,2})([pdN]))')


class SourceInt(int):
    def __new__(cls, *args, **kwargs):
        source = kwargs.pop('source', None)
        _cls = super(SourceInt, cls).__new__(cls, *args, **kwargs)
        _cls.source = source
        return _cls

    @classmethod
    def create(cls, value, source):
        if value is None:
            return None

        return cls(value, source=source)


class Baggage(object):
    def __init__(self, pieces, weight, included=None):
        # type: (Optional[SourceInt], Optional[SourceInt], Optional[SourceInt]) -> None
        self.pieces = pieces
        self.weight = weight
        self.included = included

    def __eq__(self, other):
        if not self.__class__ == other.__class__:
            raise NotImplementedError
        return (
            self.pieces == other.pieces and
            self.weight == other.weight and
            self.included == other.included
        )

    def __hash__(self):
        return hash((self.pieces, self.weight, self.included))

    def __nonzero__(self):
        return bool(self.included)

    def __repr__(self):
        if self.included is None:
            return 'None'
        else:
            weight = ' {}kg'.format(self.weight) if self.weight is not None else ''
            return "{}pc{}".format(self.pieces, weight)

    @classmethod
    def from_partner(cls, pieces=None, weight=None, included=None):
        """
        Baggage.from_partner() - no info about baggage
        Baggage.from_partner(0) - no baggage
        Baggage.from_partner(1) - 1pc with unknown weight
        Baggage.from_partner(N, 23) - Npc with 23kg
        Baggage.from_partner(weight=23) - 1pc with 23kg
        Baggage.from_partner(included=True) - included, but unknown weight and pc

        :type pieces: int|None
        :type weight: int|None
        :type included: int|bool|None
        :rtype: Baggage
        """
        pieces = int(pieces) if isinstance(pieces, basestring) else pieces
        weight = int(weight) if isinstance(weight, basestring) else weight
        if pieces is None:
            if weight == 0 or included in {False, 0}:
                pieces = 0
            elif weight is not None:
                pieces = 1
        if included is None and (pieces is not None or weight is not None):
            included = bool(pieces or weight)

        return cls(
            pieces=SourceInt.create(pieces, 'partner'),
            weight=SourceInt.create(weight, 'partner'),
            included=SourceInt.create(included, 'partner'),
        )

    @classmethod
    def create_empty(cls):
        return cls(
            included=None,
            pieces=None,
            weight=None
        )

    @classmethod
    def from_airline_tariff(cls, airline_tariff):
        # type: (Optional[CompanyTariff]) -> Baggage
        if airline_tariff is None:
            return cls.create_empty()
        return Baggage.from_db(
            included=airline_tariff.baggage_allowed,
            pieces=airline_tariff.baggage_pieces,
            weight=airline_tariff.baggage_norm,
        )

    @classmethod
    def from_db(cls, included=None, pieces=None, weight=None):
        return cls(
            included=SourceInt.create(included, 'db'),
            pieces=SourceInt.create(pieces, 'db'),
            weight=SourceInt.create(weight, 'db'),
        )

    def key(self):
        key = ''
        for field in (self.included, self.pieces, self.weight):
            key += '{0}{0.source[0]}'.format(field) if field is not None else 'N'
        return key if key != 'NNN' else None

    def as_dict(self):
        """
            {
               "included": {"count": 1, "source": 'partner'},
               "pc": {"count": 1, "source": 'db'},
               "wt": {"count": 10, "source": 'db'}
            }
        """
        baggage = {
            'included': {"count": self.included,
                         "source": self.included.source} if self.included is not None else None,
            'pc': {"count": self.pieces,
                   "source": self.pieces.source} if self.pieces is not None else None,
            'wt': {"count": self.weight,
                   "source": self.weight.source} if self.weight is not None else None,
        }
        return {
            'key': self.key(),
            'info': baggage,
        }

    @staticmethod
    def info_from_key(key):
        fields = ('included', 'pieces', 'weight')
        if not key:
            return dict.fromkeys(fields)
        baggage = {}
        for (_, count, source), field in zip(KEY_PARTS.findall(key), fields):
            baggage[field] = {
                'count': int(count),
                'source': 'partner' if source == 'p' else 'db'
            } if source != 'N' else None
        return baggage


class BaggageParser(object):
    RE_BAGGAGE = re.compile(
        r'^((?P<count>\d+)(N|PC))?((?P<weight>\d+)((?<=PC\d\d)|(K|KG)))?$',
        re.IGNORECASE
    )

    def __init__(self, logger):
        self._logger = logger

    def parse_from_string(self, baggage):
        """
        Parse basic baggage strings
        e.g. 1PC, 0PC, 2PC, 1PC23, 1PC20K, 1PC15KG, 23K, 15KG, N/A
        1pc, 1pc23kg, 1pc23k 1pc23, 23kg, 23k
        :param basestring baggage:
        :rtype: Baggage
        """
        try:
            if baggage is None or baggage == 'N/A':
                return Baggage.from_partner()

            re_result = self.RE_BAGGAGE.match(baggage)
            if re_result:
                groups = re_result.groupdict()
                count, weight = groups['count'], groups['weight']
                return Baggage.from_partner(
                    pieces=int(count) if count else None,
                    weight=int(weight) if weight else None,
                )
            else:
                self._logger.info('Unknown baggage format %r', baggage)

        except Exception as e:
            self._logger.error(
                'Baggage parsing exception: "%s" on element %r', e, baggage
            )

        return Baggage.from_partner()
