# coding=utf-8
from logging import getLogger

from travel.avia.library.python.avia_data.models import AviaCompany
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.utils import environment

from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage, SourceInt
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.yt_logger import YtLogger


class BaggageMerger(object):
    def __init__(self, logger, yt_logger):
        self._logger = logger
        self._yt_logger = yt_logger

    def _safe_merge(self, airline, default_baggage, partner_baggage):
        # Если у нас нет информации о багажной информации авиакомпании
        # То недоверяем инфорации от партнера совсем
        if not airline:
            return Baggage.create_empty()

        try:
            airline_type = airline.cost_type
            if airline_type == AviaCompany.NORMAL_COST:
                return self._merge_for_normal(default_baggage, partner_baggage)
            if airline_type == AviaCompany.HYBRID_COST:
                return self._merge_for_hybrid(default_baggage, partner_baggage)
            if airline_type == AviaCompany.LOW_COST:
                return self._merge_for_low_cost()

            raise Exception('unknown airline type: %r', airline_type)
        except Exception as e:
            self._logger.exception('BaggageMerger fail: %r', e)
            return Baggage.create_empty()

    def merge(self, partner_code, flight_number, airline, default_baggage, partner_baggage, company, is_charter):
        if is_charter and partner_baggage is not None:
            return partner_baggage
        if partner_baggage is None:
            partner_baggage = Baggage.create_empty()

        merged_baggage = self._safe_merge(
            airline=airline, default_baggage=default_baggage, partner_baggage=partner_baggage
        )

        self._log(
            partner_code=partner_code,
            flight_number=flight_number,
            company=company,
            airline=airline,
            partner_baggage=partner_baggage,
            default_baggage=default_baggage,
            merged_baggage=merged_baggage,
        )

        return merged_baggage

    def merge_variant_baggage(self, v):
        baggage_list = [s.baggage for s in v.forward.segments + v.backward.segments]
        has_baggage = all(b and b.included for b in baggage_list)
        if has_baggage:
            return Baggage(
                included=v.forward.segments[0].baggage.included,
                weight=min(b.weight for b in baggage_list),
                pieces=min(b.pieces for b in baggage_list),
            )

        return Baggage(
            included=SourceInt(0, source='db'),
            weight=SourceInt(0, source='db'),
            pieces=SourceInt(0, source='db'),
        )

    def _merge_for_normal(self, default_baggage, partner_baggage):
        #  Тут мы считаем, что в нашей базе всегда правда

        if default_baggage.included and (partner_baggage.pieces > 0):
            pieces = partner_baggage.pieces
        else:
            pieces = default_baggage.pieces

        return Baggage(
            included=default_baggage.included if default_baggage.included is not None else SourceInt(1, source='db'),
            pieces=pieces,
            weight=default_baggage.weight,
        )

    def _merge_for_hybrid(self, default_baggage, partner_baggage):
        if partner_baggage.included is None:
            return Baggage(
                included=default_baggage.included,
                pieces=default_baggage.pieces,
                weight=default_baggage.weight,
            )

        return Baggage(
            included=partner_baggage.included,
            pieces=partner_baggage.pieces,
            weight=(SourceInt(0, source='partner') if partner_baggage.included == 0 else default_baggage.weight),
        )

    def _merge_for_low_cost(self):
        return Baggage(
            included=SourceInt(0, source='db'), pieces=SourceInt(0, source='db'), weight=SourceInt(0, source='db')
        )

    def _dump_source(self, source_int):
        if source_int is None:
            return None

        return source_int.source

    def _log(self, partner_code, flight_number, company, airline, partner_baggage, default_baggage, merged_baggage):
        # type: (str, str, Company, AviaCompany, Baggage, Baggage, Baggage) -> None

        airline_id = None
        cost_type = None
        company_id = None
        if airline:
            airline_id = airline.rasp_company_id
            cost_type = airline.cost_type
        if company:
            company_id = company.id

        self._yt_logger.log(
            {
                'partner': partner_code,
                'flight_number': flight_number,
                'airline_id': airline_id,
                'airline_cost_type': cost_type,
                'company_id': company_id,
                'included_by_partner': partner_baggage.included,
                'pieces_by_partner': partner_baggage.pieces,
                'weight_by_partner': partner_baggage.weight,
                'included_by_db': default_baggage.included,
                'pieces_by_db': default_baggage.pieces,
                'weight_by_db': default_baggage.weight,
                'merged_included_value': merged_baggage.included,
                'merged_pieces_value': merged_baggage.pieces,
                'merged_weight_value': merged_baggage.weight,
                'merged_included_source': self._dump_source(merged_baggage.included),
                'merged_pieces_source': self._dump_source(merged_baggage.pieces),
                'merged_weight_source': self._dump_source(merged_baggage.weight),
            }
        )


baggage_merger = BaggageMerger(
    logger=getLogger(__name__), yt_logger=YtLogger('yt.baggage_info', environment=environment)
)
